import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.Iterator;

/**
 * Memory used to cache data that resides on disk.
 * 
 * @author Michael D. Naper, Jr. <MichaelNaper.com>
 * @version 2013.01.17
 */
public class BufferPool {

  // File where data resides
  private final RandomAccessFile dataFile;

  // Pool of currently allocated buffers in order of last access
  private final Deque<Buffer> pool;

  // Maximum number of buffers to be allocated
  private final int capacity;

  // Maximum number of bytes each buffer can hold (data file split into blocks
  // of this size)
  private final int bufferSize;

  // Number of cache hits
  private int numCacheHits;

  // Number of cache misses
  private int numCacheMisses;

  // Number of disk reads
  private int numDiskReads;

  // Number of disk writes
  private int numDiskWrites;

  /**
   * Constructs a new, empty {@code BufferPool} with the specified data file,
   * capacity, and block size.
   * 
   * @param dataFile
   *          The file where the data that is to be cached resides.
   * @param capacity
   *          The maximum number of buffers to be allocated.
   * @param bufferSize
   *          The maximum number of bytes each buffer can hold. The data file is
   *          split into blocks of this size.
   * @throws FileNotFoundException
   *           Thrown if the file exists but is a directory rather than a
   *           regular file, or cannot be opened or created for any other
   *           reason.
   */
  public BufferPool(File dataFile, int capacity, int bufferSize)
      throws FileNotFoundException {
    if (dataFile == null) {
      throw new NullPointerException("dataFile is null.");
    }
    if (capacity <= 0) {
      throw new IllegalArgumentException("capacity must be a positive integer.");
    }
    if (bufferSize <= 0) {
      throw new IllegalArgumentException(
          "bufferSize must be a positive integer.");
    }

    this.dataFile = new RandomAccessFile(dataFile, "rw");
    this.capacity = capacity;
    this.bufferSize = bufferSize;
    pool = new ArrayDeque<Buffer>(capacity);
    numCacheHits = numCacheMisses = numDiskReads = numDiskWrites = 0;
  }

  /**
   * Reads and retrieves the data at the specified file offset position,
   * measured from the beginning of the data file, up to the specified length in
   * bytes.
   * 
   * @param pos
   *          The offset position, measured in bytes from the beginning of the
   *          data file, at which to begin reading data.
   * @param length
   *          The number of bytes to read.
   * @throws IOException
   *           If an I/O error occurs.
   */
  public byte[] read(int pos, int length) throws IOException {
    if (pos < 0) {
      throw new IllegalArgumentException("pos must be a non-negative integer.");
    }
    if (length <= 0) {
      throw new IllegalArgumentException("length must be a positive integer.");
    }
    if (pos + length > dataFile.length()) {
      throw new IllegalArgumentException("File out of bounds. [pos: " + pos
          + ", length: " + length + ", file length: " + dataFile.length() + "]");
    }

    byte[] data = new byte[length];
    int blockId = pos / bufferSize;
    int blockOffset = pos % bufferSize;
    for (int dataPos = 0; dataPos < length;) {
      Buffer buffer = getBuffer(blockId++);
      dataPos += buffer.read(blockOffset, data, dataPos);
      blockOffset = 0;
    }
    return data;
  }

  /**
   * Writes the specified data to the data file beginning at the specified file
   * offset position, measured from the beginning of the data file.
   * 
   * @param data
   *          The data to write to the data file.
   * @param pos
   *          The offset position, measured in bytes from the beginning of the
   *          data file, at which to begin writing data.
   * @throws IOException
   *           If an I/O error occurs.
   */
  public void write(byte[] data, int pos) throws IOException {
    if (data == null) {
      throw new NullPointerException("data is null.");
    }
    if (pos < 0) {
      throw new IllegalArgumentException("pos must be a non-negative integer.");
    }

    int blockId = pos / bufferSize;
    int blockOffset = pos % bufferSize;
    for (int dataPos = 0; dataPos < data.length;) {
      Buffer buffer = getBuffer(blockId++);
      dataPos += buffer.write(blockOffset, data, dataPos);
      blockOffset = 0;
    }

    if (pos + data.length > dataFile.length()) {
      dataFile.setLength(pos + data.length);
    }
  }

  /**
   * Flushes this buffer pool, writing all cached data back to the data file.
   * 
   * @throws IOException
   *           If an I/O error occurs.
   */
  public void flush() throws IOException {
    for (Buffer buffer : pool) {
      flushBuffer(buffer);
    }
  }

  /**
   * Returns the {@link Buffer} storing the data at the file block corresponding
   * to the specified ID.
   * 
   * @param id
   *          The ID of the file block whose containing buffer is to be
   *          returned.
   * @return The buffer storing the data at the file block corresponding to the
   *         specified ID.
   * @throws IOException
   *           If an I/O error occurs.
   */
  private Buffer getBuffer(int id) throws IOException {
    assert id >= 0 : "id must be a non-negative integer.";

    // Find block if it has already been cached
    Iterator<Buffer> iter = pool.iterator();
    while (iter.hasNext()) {
      Buffer buffer = iter.next();
      if (buffer.id == id) {
        numCacheHits++;
        iter.remove();
        pool.offerFirst(buffer);
        return buffer;
      }
    }

    // Cache block, replacing a buffer if necessary
    numCacheMisses++;
    if (pool.size() == capacity) {
      replaceBuffer();
    }
    Buffer newBuffer = allocateBuffer(id);
    pool.offerFirst(newBuffer);
    return newBuffer;
  }

  /**
   * Allocates and returns a new {@link Buffer} storing the data at the
   * specified file block.
   * 
   * @param id
   *          The ID of the block to store in the buffer.
   * @return The newly allocated buffer storing the data at the specified file
   *         block.
   * @throws IOException
   *           If an I/O error occurs.
   */
  private Buffer allocateBuffer(int id) throws IOException {
    assert id >= 0 : "id must be a non-negative integer.";

    numDiskReads++;
    int pos = id * bufferSize;
    dataFile.seek(pos);
    byte[] block = new byte[bufferSize];
    int size = dataFile.read(block);
    return new Buffer(id, block, size);
  }

  /**
   * Replaces the least recently accessed {@link Buffer} in the pool.
   * 
   * @throws IOException
   *           If an I/O error occurs.
   */
  private void replaceBuffer() throws IOException {
    Buffer replacedBuffer = pool.pollLast();
    flushBuffer(replacedBuffer);
  }

  /**
   * Flushes the specified {@link Buffer}, writing the block of data contained
   * in the buffer back to its corresponding location in the data file.
   * 
   * @param buffer
   *          The buffer to be flushed.
   * @throws IOException
   *           If an I/O error occurs.
   */
  private void flushBuffer(Buffer buffer) throws IOException {
    if (buffer.isDirty()) {
      numDiskWrites++;
      int pos = buffer.id * bufferSize;
      dataFile.seek(pos);
      dataFile.write(buffer.block, 0, buffer.size);
    }
  }

  /**
   * Returns the length of the data file.
   * 
   * @return The length of the data file, measured in bytes.
   * @throws IOException
   *           If an I/O error occurs.
   */
  public long fileLength() throws IOException {
    return dataFile.length();
  }

  /**
   * Truncates the length of the data file to the specified length.
   * 
   * @param length
   *          The desired length of the file.
   * @throws IOException
   *           If an I/O error occurs.
   */
  public void truncateFile(long length) throws IOException {
    if (length > dataFile.length()) {
      throw new IllegalArgumentException(
          "length must be less than or equal to the current length of the file.");
    }

    dataFile.setLength(length);

    // Update buffers to reflect file length
    int maxBlockId = (int) (length / bufferSize);
    int maxBlockSize = (int) (length % bufferSize);
    if (maxBlockSize == 0) {
      maxBlockId--;
      maxBlockSize = bufferSize;
    }
    Iterator<Buffer> iter = pool.iterator();
    while (iter.hasNext()) {
      Buffer buffer = iter.next();
      if (buffer.id == maxBlockId) {
        buffer.size = maxBlockSize;
      } else if (buffer.id > maxBlockId) {
        iter.remove();
      }
    }
  }

  /**
   * Returns the size of a file block. That is, returns the maximum number of
   * bytes each buffer in this buffer pool can hold.
   * 
   * @return The size of a file block.
   */
  public int getBlockSize() {
    return bufferSize;
  }

  /**
   * Closes this buffer pool.
   * 
   * @throws IOException
   *           Thrown if an I/O error occurs.
   */
  public void close() throws IOException {
    dataFile.close();
  }

  /**
   * Returns the number of cache hits. That is, returns the number of times
   * requested data was stored in a buffer and did not have to be read from
   * disk.
   * 
   * @return The number of cache hits.
   */
  public int getNumCacheHits() {
    return numCacheHits;
  }

  /**
   * Returns the number of cache misses. That is, returns the number of times
   * requested data was not stored in a buffer and had to be read from disk.
   * 
   * @return The number of cache misses.
   */
  public int getNumCacheMisses() {
    return numCacheMisses;
  }

  /**
   * Returns the number of disk reads. That is, returns the number of times a
   * block of data was read from disk and stored in a buffer.
   * 
   * @return The number of disk reads.
   */
  public int getNumDiskReads() {
    return numDiskReads;
  }

  /**
   * Returns the number of disk writes. That is, returns the number of times a
   * block of data stored in a buffer was written to disk.
   * 
   * @return The number of disk writes.
   */
  public int getNumDiskWrites() {
    return numDiskWrites;
  }

  /**
   * Returns a string of the block IDs currently stored by this buffer pool in
   * order from most recently accessed to least recently accessed.
   * 
   * @return A string of the block IDs currently stored by this buffer pool in
   *         order from most recently accessed to least recently accessed.
   */
  public String blockIDs() {
    StringBuilder sb = new StringBuilder();
    Iterator<Buffer> iter = pool.iterator();
    while (iter.hasNext()) {
      sb.append(iter.next().id);
      if (iter.hasNext()) {
        sb.append(", ");
      }
    }
    return sb.toString();
  }

  /**
   * Represents a buffer for storing a block of data from the data file.
   */
  private static final class Buffer {

    // ID of the block stored by this buffer
    final int id;

    // Block of data
    final byte[] block;

    // Number of bytes stored by this buffer
    int size;

    // Indicates if the block has been modified
    boolean dirtyBit;

    /**
     * Constructs a new {@code Buffer} with the specified ID of the block of the
     * file this buffer represents, the space storing the unmodified block of
     * data, and the size of the block of data.
     * 
     * @param id
     *          The ID of the block stored by this buffer.
     * @param block
     *          The unmodified block of data.
     * @param size
     *          The size of the block of data.
     */
    Buffer(int id, byte[] block, int size) {
      assert id >= 0 : "id must be a non-negative integer.";
      assert block != null : "block is null.";
      assert size > 0 : "size must be a non-negative integer.";
      assert size < block.length : "size must be less than the length of the block.";

      this.id = id;
      this.block = block;
      this.size = size;
      dirtyBit = false;
    }

    /**
     * Reads the block of data contained in this buffer beginning at the
     * specified offset into the specified array of bytes starting at the
     * specified position. Reads up to the length of the specified byte array or
     * the size of this buffer.
     * 
     * @param offset
     *          The offset of the block of data contained in this buffer at
     *          which to begin reading bytes.
     * @param bytes
     *          The buffer into which the bytes are read.
     * @param bytesPos
     *          The starting position in the specified byte array into which to
     *          begin reading.
     * @return The number of bytes read from the block of data contained in this
     *         buffer.
     */
    int read(int offset, byte[] bytes, int bytesPos) {
      assert offset >= 0 : "offset must be a non-negative integer.";
      assert offset < block.length : "offset must be less than the length of the data block.";
      assert bytes != null : "bytes is null.";
      assert bytesPos >= 0 : "bytesPos must be a non-negative integer.";
      assert bytesPos < bytes.length : "bytesPos must be less than the length of the specified bytes.";

      int numBytesRead = Math.min(bytes.length - bytesPos, size - offset);
      System.arraycopy(block, offset, bytes, bytesPos, numBytesRead);
      return numBytesRead;
    }

    /**
     * Writes the bytes from the specified byte array starting at the specified
     * position to the block of data contained in this buffer beginning at the
     * specified offset. Writes up to the length of the specified byte array or
     * the size of this buffer.
     * 
     * @param offset
     *          The offset of the block of data contained in this buffer at
     *          which to begin writing the bytes.
     * @param bytes
     *          The byte array storing the bytes to be written.
     * @param bytesPos
     *          The starting position in the specified byte array from which to
     *          begin writing.
     * @return The number of bytes written to the block of data contained in
     *         this buffer.
     */
    int write(int offset, byte[] bytes, int bytesPos) {
      assert offset >= 0 : "offset must be a non-negative integer.";
      assert offset < block.length : "offset must be less than the length of the data block.";
      assert bytes != null : "bytes is null.";
      assert bytesPos >= 0 : "bytesPos must be a non-negative integer.";
      assert bytesPos < bytes.length : "bytesPos must be less than the length of the specified bytes.";

      int numBytesWritten = Math.min(bytes.length - bytesPos, block.length
          - offset);
      System.arraycopy(bytes, bytesPos, block, offset, numBytesWritten);
      dirtyBit = true;
      if (offset + numBytesWritten > size) {
        size = offset + numBytesWritten;
      }
      return numBytesWritten;
    }

    /**
     * Returns {@code true} if the block of data contained in this
     * {@code Buffer} has been modified.
     * 
     * @return {@code true} if the block of data contained in this
     *         {@code Buffer} has been modified.
     */
    boolean isDirty() {
      return dirtyBit;
    }
  }
}
