import java.util.List;
import java.util.ListIterator;
import java.util.NoSuchElementException;

/**
 * An iterator for lists that allows the programmer to traverse the list with
 * behavior identical to that of {@link ListIterator} but which treats the list
 * as if it were circular. Specifically, the iterator will treat the first
 * element as succeeding the last element, and the last element as preceding the
 * first element.
 * <p>
 * Add operations made at either end of the list using this iterator modifies
 * the underlying list based on the cursor position as it would be for a
 * standard {@code ListIterator}. That is, if the first element of the list was
 * the last element returned by a call to {@code previous}, a call to
 * {@code add} will insert the element at the front of the list; likewise, if
 * the last element of the list was the last element returned by a call to
 * {@code next}, a call to {@code add} will insert the element at the rear of
 * the list.
 * 
 * @author Michael D. Naper, Jr. <MichaelNaper.com>
 * @version 2013.01.17
 * @param <E>
 *          The type of elements in the list being traversed.
 * @see List
 */
public class CircularListIterator<E> implements ListIterator<E> {

  // List underlying iterator
  private final List<E> list;

  // Iterator currently iterating over the list
  private ListIterator<E> iter;

  public CircularListIterator(List<E> list) {
    this(list, 0);
  }

  /**
   * Constructs a new {@code CircularListIterator} over the elements in the
   * specified list, starting at the specified position in the list.
   * 
   * @param list
   *          The list whose elements are to be iterated over.
   * @param index
   *          The index of the first element to be returned from the iterator
   *          (by a call to the {@link #next} method).
   */
  public CircularListIterator(List<E> list, int index) {
    if (list == null) {
      throw new NullPointerException("list is null.");
    }

    this.list = list;
    iter = list.listIterator(index);
  }

  @Override
  public int nextIndex() {
    return iter.hasNext() ? iter.nextIndex() : 0;
  }

  @Override
  public boolean hasNext() {
    return list.size() > 0;
  }

  @Override
  public E next() {
    if (!hasNext()) {
      throw new NoSuchElementException();
    }
    if (!iter.hasNext()) {
      iter = list.listIterator();
    }
    return iter.next();
  }

  @Override
  public int previousIndex() {
    if (iter.hasPrevious()) {
      return iter.previousIndex();
    } else {
      return list.size() > 0 ? list.size() - 1 : 0;
    }
  }

  @Override
  public boolean hasPrevious() {
    return list.size() > 0;
  }

  @Override
  public E previous() {
    if (!hasPrevious()) {
      throw new NoSuchElementException();
    }
    if (!iter.hasPrevious()) {
      iter = list.listIterator(list.size());
    }
    return iter.previous();
  }

  @Override
  public void add(E element) {
    iter.add(element);
  }

  @Override
  public void set(E element) {
    iter.set(element);
  }

  @Override
  public void remove() {
    iter.remove();
  }
}
