/**
 * A command representing an action to be taken.
 * 
 * @author Michael D. Naper, Jr. <MichaelNaper.com>
 * @version 2013.01.17
 */
public abstract class Command {

  /**
   * Type of this command.
   */
  protected final Type type;

  /**
   * Constructs a new {@code Command} with the specified {@link Type}.
   * 
   * @param type
   *          The type of this command.
   */
  public Command(Type type) {
    this.type = type;
  }

  /**
   * Returns the {@link Type} of this command.
   * 
   * @return The type of this command.
   */
  public Type getType() {
    return type;
  }

  /**
   * Executes this command, invoking actions upon the specified
   * {@link Controller}.
   * 
   * @param controller
   *          The {@code Controller} upon which to invoke actions.
   */
  public abstract void execute(Controller controller);

  /**
   * Represents all valid command types.
   */
  public enum Type {
    INSERT, REMOVE, SEARCH, PRINT;
  }

  /**
   * Represents a no-operation command.
   */
  public static class NullCommand extends Command {

    /**
     * Constructs a new {@code NullCommand} whose {@link Type} is {@code null}.
     */
    public NullCommand() {
      super(null);
    }

    @Override
    public void execute(Controller controller) {
      if (controller == null) {
        throw new NullPointerException("controller is null.");
      }

      // nothing to execute
    }
  }
}
