import java.io.File;
import java.io.FileNotFoundException;
import java.util.NoSuchElementException;
import java.util.Scanner;

/**
 * A parser for parsing the commands specified in a command file.
 * <p>
 * This parser breaks its input into commands using line delimiters.
 * 
 * @author Michael D. Naper, Jr. <MichaelNaper.com>
 * @version 2013.01.17
 */
public class CommandParser {

  // Scanner for parsing the command file line-by-line
  private final Scanner lineScanner;

  // Buffer used to hold next command, or null if no command exists
  private String nextCommand;

  // Flag indicating whether this parser is closed
  private boolean closed;

  /**
   * Constructs a new {@code CommandParser} with the specified file from which
   * to parse commands.
   * 
   * @param commandFile
   *          The command file from which to parse commands.
   * @throws FileNotFoundException
   *           If the specified command file is not found.
   */
  public CommandParser(File commandFile) throws FileNotFoundException {
    if (commandFile == null) {
      throw new NullPointerException("commandFile is null.");
    }

    this.lineScanner = new Scanner(commandFile);
  }

  /**
   * Returns {@code true} if this parser has another command to parse from the
   * command file.
   * 
   * @return {@code true} if and only if this parser has another command to
   *         parse.
   * @throws IllegalStateException
   *           If this parser is closed.
   */
  public boolean hasCommand() {
    ensureOpen();
    jumpToNextCommand();
    return nextCommand != null;
  }

  /**
   * Moves the scanner to the next non-blank line in the command file or to the
   * end of the file if no such line exists. Upon return, {@code nextCommand}
   * will store the next command line, or {@code null} if the end of file has
   * been reached.
   */
  private final void jumpToNextCommand() {
    if (nextCommand == null) {
      while (lineScanner.hasNextLine()) {
        nextCommand = lineScanner.nextLine().trim();
        if (!nextCommand.isEmpty()) {
          return; // next non-blank line has been found
        }
      }
      nextCommand = null; // end of file has been reached
    }
  }

  /**
   * Parses the next command in the command file.
   * 
   * @return A {@link Command} representing the parsed command.
   * @throws InvalidCommandException
   *           If an invalid command is attempting to be parsed.
   * @throws NoSuchElementException
   *           If no more commands are available to be parsed.
   * @throws IllegalStateException
   *           If this parser is closed.
   */
  public Command nextCommand() throws InvalidCommandException {
    ensureOpen();
    if (!hasCommand()) {
      throw new NoSuchElementException("End of command file has been reached.");
    }

    Scanner commandScanner = new Scanner(nextCommand);
    nextCommand = null;

    String commandTypeString = commandScanner.next();
    Command.Type commandType;
    try {
      commandType = Command.Type.valueOf(commandTypeString.toUpperCase());
    } catch (IllegalArgumentException e) {
      commandScanner.close();
      throw new InvalidCommandException(commandTypeString
          + " is not a valid command type.");
    }

    Command command;
    switch (commandType) {
      case INSERT:
        command = parseInsertCommand(commandScanner);
        break;
      case REMOVE:
        command = parseRemoveCommand(commandScanner);
        break;
      case SEARCH:
        command = parseSearchCommand(commandScanner);
        break;
      case PRINT:
        command = parsePrintCommand(commandScanner);
        break;
      default:
        command = new Command.NullCommand();
    }
    commandScanner.close();
    return command;
  }

  /**
   * Ensures that this parser is open. Throws an {@code IllegalStateException}
   * if the parser is closed.
   */
  private void ensureOpen() {
    if (closed) {
      throw new IllegalStateException("Command parser is closed.");
    }
  }

  /**
   * Parses a command from the specified scanner as an {@link InsertCommand}.
   * 
   * @param lineScanner
   *          The scanner from which to parse.
   * @return An {@code InsertCommand} representing the parsed command.
   * @throws InvalidCommandException
   *           If the next sequence of tokens do not match the expression for an
   *           {@code InsertCommand}.
   */
  private InsertCommand parseInsertCommand(Scanner lineScanner)
      throws InvalidCommandException {
    assert lineScanner != null : "lineScanner is null.";

    try {
      String dnaString = lineScanner.next();
      return new InsertCommand(dnaString);
    } catch (NoSuchElementException e) {
      lineScanner.close();
      throw new InvalidCommandException(
          "Insert command arguments must take the form: " + "\"<DNA-string>\"");
    }
  }

  /**
   * Parses a command from the specified scanner as a {@link RemoveCommand}.
   * 
   * @param lineScanner
   *          The scanner from which to parse.
   * @return A {@code RemoveCommand} representing the parsed command.
   * @throws InvalidCommandException
   *           If the next sequence of tokens do not match the expression for a
   *           {@code RemoveCommand}.
   */
  private RemoveCommand parseRemoveCommand(Scanner lineScanner)
      throws InvalidCommandException {
    assert lineScanner != null : "lineScanner is null.";

    try {
      String dnaString = lineScanner.next();
      return new RemoveCommand(dnaString);
    } catch (NoSuchElementException e) {
      lineScanner.close();
      throw new InvalidCommandException(
          "Remove command arguments must take the form: " + "\"<DNA-string>\"");
    }
  }

  /**
   * Parses a command from the specified scanner as a {@link SearchCommand}.
   * 
   * @param lineScanner
   *          The scanner from which to parse.
   * @return A {@code SearchCommand} representing the parsed command.
   * @throws InvalidCommandException
   *           If the next sequence of tokens do not match the expression for a
   *           {@code SearchCommand}.
   */
  private SearchCommand parseSearchCommand(Scanner lineScanner)
      throws InvalidCommandException {
    assert lineScanner != null : "lineScanner is null.";

    try {
      String dnaStringDescriptor = lineScanner.next();
      return new SearchCommand(dnaStringDescriptor);
    } catch (NoSuchElementException e) {
      lineScanner.close();
      throw new InvalidCommandException(
          "Search command arguments must take the form: "
              + "\"<DNA-string-descriptor>\"");
    }
  }

  /**
   * Parses a command from the specified scanner as a {@link PrintCommand}.
   * 
   * @param lineScanner
   *          The scanner from which to parse.
   * @return A {@code PrintCommand} representing the parsed command.
   * @throws InvalidCommandException
   *           If the next sequence of tokens do not match the expression for a
   *           {@code PrintCommand}.
   */
  private PrintCommand parsePrintCommand(Scanner lineScanner)
      throws InvalidCommandException {
    assert lineScanner != null : "lineScanner is null.";

    PrintCommand.PrintType type;
    if (!lineScanner.hasNext()) {
      type = PrintCommand.PrintType.DEFAULT;
    } else {
      String printType = lineScanner.next();
      if (PrintCommand.PrintType.LENGTHS.toString().equals(printType)) {
        type = PrintCommand.PrintType.LENGTHS;
      } else if (PrintCommand.PrintType.STATISTICS.toString().equals(printType)) {
        type = PrintCommand.PrintType.STATISTICS;
      } else {
        lineScanner.close();
        throw new InvalidCommandException(
            "Print command arguments must take the form: " + "\"[<type>]\"");
      }
    }
    return new PrintCommand(type);
  }

  /**
   * Closes this command file scanner.
   */
  public void close() {
    lineScanner.close();
    closed = true;
  }

  /**
   * Thrown to indicate that an invalid command has been encountered.
   */
  public static class InvalidCommandException extends Exception {

    private static final long serialVersionUID = 5910093037795852176L;

    /**
     * Constructs a new {@code InvalidCommandException}.
     */
    public InvalidCommandException() {
      super();
    }

    /**
     * Constructs a new {@code InvalidCommandException} with the specified
     * detail message.
     * 
     * @param message
     *          The detail message.
     */
    public InvalidCommandException(String message) {
      super(message);
    }
  }
}
