import java.io.PrintWriter;
import java.util.Collection;

/**
 * A controller that manages the retrieval of commands and makes the necessary
 * calls to the other system components in order to carry out those commands.
 * 
 * @author Michael D. Naper, Jr. <MichaelNaper.com>
 * @version 2013.01.17
 */
public class Controller {

  // Record database for storing and managing DNA string records
  private final RecordDatabase recordDatabase;

  // Memory manager for storing record data
  private final MemoryManager memoryManager;

  // Buffer pool buffering the file in which the record data are stored
  private final BufferPool bufferPool;

  // Parser for parsing the command file
  private final CommandParser commandParser;

  // Print writer for printing output
  private final PrintWriter printWriter;

  /**
   * Constructs a new {@code Controller} with the specified
   * {@link RecordDatabase}, {@link MemoryManager}, {@link BufferPool}, and
   * {@link CommandParser} and prints output using the specified print writer.
   * 
   * @param recordDatabase
   *          The record manager for storing and managing records.
   * @param memoryManager
   *          The memory manager for storing record data.
   * @param bufferPool
   *          The buffer pool buffering the file in which the record data are
   *          stored.
   * @param commandParser
   *          The command parser from which to retrieve commands.
   * @param printWriter
   *          The print writer to be used for printing output.
   */
  public Controller(RecordDatabase recordDatabase, MemoryManager memoryManager,
      BufferPool bufferPool, CommandParser commandParser,
      PrintWriter printWriter) {
    this.recordDatabase = recordDatabase;
    this.memoryManager = memoryManager;
    this.bufferPool = bufferPool;
    this.commandParser = commandParser;
    this.printWriter = printWriter;

    if (this.recordDatabase == null) {
      throw new NullPointerException("recordManager is null.");
    }
    if (this.memoryManager == null) {
      throw new NullPointerException("memoryManager is null.");
    }
    if (this.bufferPool == null) {
      throw new NullPointerException("bufferPool is null.");
    }
    if (this.commandParser == null) {
      throw new NullPointerException("commandParser is null.");
    }
    if (this.printWriter == null) {
      throw new NullPointerException("printWriter is null.");
    }
  }

  /**
   * Begins fetching the commands sequentially from the command file and
   * executing the appropriate actions.
   */
  public void execute() {
    printWriter.println("CS 3114: Data Structures and Algorithms");
    printWriter.println("Project 4: DNA File\n");
    printWriter.println("Author: Michael D. Naper, Jr.");
    printWriter.println("-----------------------------");

    for (int commandNumber = 1; commandParser.hasCommand(); commandNumber++) {
      printWriter.print("\nCommand " + commandNumber + ": ");
      Command command;
      try {
        command = commandParser.nextCommand();
      } catch (CommandParser.InvalidCommandException e) {
        System.err.println(e.getMessage());
        continue;
      }
      command.execute(this);
    }
  }

  /**
   * Inserts the specified {@link DnaString} record into the record database if
   * it is not already present.
   * 
   * @param record
   *          The record to be inserted.
   */
  public void insertRecord(DnaString record) {
    if (record == null) {
      throw new NullPointerException("record is null.");
    }

    printWriter.print(Command.Type.INSERT + " " + record + "\n");

    try {
      int insertDepth = recordDatabase.insert(record);
      if (insertDepth < 0) {
        printWriter.println("insert FAILED");
        return;
      }
      printWriter.println("inserted --> " + record + " [Depth: " + insertDepth
          + "]");
    } catch (IllegalArgumentException e) {
      System.err.println(e.getMessage());
      printWriter.println("insert FAILED");
      return;
    }
  }

  /**
   * Removes the specified {@link DnaString} record from the record database if
   * it is present.
   * 
   * @param record
   *          The record to be removed, if present.
   */
  public void removeRecord(DnaString record) {
    printWriter.print(Command.Type.REMOVE + " " + record + "\n");

    printWriter.print("removed --> ");
    if (recordDatabase.remove(record)) {
      printWriter.println(record);
    } else {
      printWriter.println("[no record]");
    }
  }

  /**
   * Searches for and prints all {@link DnaString} records contained in the
   * record database matching the specified {@link DnaStringDescriptor}.
   * 
   * @param dnaStringDescriptor
   *          The {@link DnaStringDescriptor} against which to match.
   */
  public void searchRecords(DnaStringDescriptor dnaStringDescriptor) {
    printWriter.print(Command.Type.SEARCH + " " + dnaStringDescriptor);
    printWriter.print("\n");

    Collection<DnaString> found = new LinkedList<>();
    int numNodesVisited = recordDatabase.search(dnaStringDescriptor, found);
    if (!found.isEmpty()) {
      for (DnaString dnaString : found) {
        printWriter.println(dnaString);
      }
    } else {
      printWriter.println("[no records]");
    }
    printWriter.println("[Visited " + numNodesVisited + " nodes in DNA tree.]");
  }

  /**
   * Dumps out a complete listing of the contents of the underlying
   * {@link MemoryPool}. This listing contains two parts: a listing of the
   * {@link DnaString} records currently stored in the memory pool, in a
   * pre-order traversal of the database's underlying {@link DnaTreeSet}, along
   * with their position in the memory pool; a listing of the blocks in the
   * {@link FreeList}.
   */
  public void dumpContents() {
    printWriter.print(Command.Type.PRINT + "\n");

    dumpContents(recordDatabase.dnaTreeToString());
  }

  /**
   * Dumps out a complete listing of the contents of the underlying
   * {@link MemoryPool}. This listing contains two parts: a listing of the
   * {@link DnaString} records currently stored in the memory pool, in a
   * pre-order traversal of the database's underlying {@link DnaTreeSet}, along
   * with their length and their position in the memory pool; a listing of the
   * blocks in the {@link FreeList}.
   */
  public void dumpContentsWithLengths() {
    printWriter.print(Command.Type.PRINT + " " + PrintCommand.PrintType.LENGTHS
        + "\n");

    dumpContents(recordDatabase.dnaTreeToStringWithLengths());
  }

  /**
   * Dumps out a complete listing of the contents of the underlying
   * {@link MemoryPool}. This listing contains two parts: a listing of the
   * {@link DnaString} records currently stored in the memory pool, in a
   * pre-order traversal of the database's underlying {@link DnaTreeSet}, along
   * with their statistics and their position in the memory pool; a listing of
   * the blocks in the {@link FreeList}.
   */
  public void dumpContentsWithStatistics() {
    printWriter.print(Command.Type.PRINT + " "
        + PrintCommand.PrintType.STATISTICS + "\n");

    dumpContents(recordDatabase.dnaTreeToStringWithStatistics());
  }

  /**
   * Dumps out a complete listing of the contents of the underlying
   * {@link MemoryPool}. This listing contains two parts: the specified string
   * representing a listing of the {@link DnaString} records currently stored in
   * the memory pool, in a pre-order traversal of the database's underlying
   * {@link DnaTreeSet}, along with their position in the memory pool; a listing
   * of the blocks in the {@link FreeList}.
   * 
   * @param dnaTreeString
   *          A string representing a listing of the records currently stored in
   *          the memory pool, in a pre-order traversal of the database's
   *          underlying {@code DnaTreeSet}, along with their position in the
   *          memory pool.
   */
  private void dumpContents(String dnaTreeString) {
    assert dnaTreeString != null : "dnaTreeString is null.";

    String divider = "-----------------------------";
    StringBuilder sb = new StringBuilder();
    sb.append(divider);
    sb.append("\nDNA Tree:\n");
    sb.append(dnaTreeString);
    sb.append("\nFree Memory Blocks:\n");
    sb.append(memoryManager.freeListToString());
    sb.append("\n\nBuffer Pool Block IDs:\n");
    sb.append(bufferPool.blockIDs());
    sb.append("\n");
    sb.append(divider);
    printWriter.println(sb.toString());
  }
}
