import java.util.Arrays;
import java.util.Collection;

/**
 * A tree based set which contains DNA strings. This set stores DNA strings with
 * a common prefix under the same sequence of edges in the tree. No node in the
 * tree stores the DNA string associated with that node; instead, its position
 * in the tree defines the DNA string with which it is associated. All the
 * descendants of a node have a common prefix of the DNA string associated with
 * that node, and the root is associated with the empty string. The DNA strings
 * are associated only with leaves.
 * <p>
 * This implementation stores the actual DNA strings on disk and maintains their
 * corresponding {@link MemoryHandle}s in the tree.
 * 
 * @author Michael D. Naper, Jr. <MichaelNaper.com>
 * @version 2013.01.17
 */
public class DiskDnaTreeSet extends DnaTreeSet {

  // Serializer for nodes
  private final Serializer<Node> NODE_SERIALIZER = new NodeSerializer();

  // Memory manager for storing and retrieving serialized DNA strings
  private final MemoryManager memoryManager;

  // Serializer for DNA strings
  private final Serializer<DnaString> dnaStringSerializer;

  // Handle to the root of this DNA tree, or null if the root is a flyweight
  // (Note: hides the super class's root reference to a Node, which is unused)
  private MemoryHandle root;

  /**
   * Constructs a new, empty {@code DiskDnaTreeSet} backed by the specified
   * {@link MemoryManager} and with the specified {@link Serializer} for
   * instances of DNA strings.
   * 
   * @param memoryManager
   *          The memory manager for storing record data.
   * @param dnaStringSerializer
   *          A serializer for instances of DNA strings.
   */
  public DiskDnaTreeSet(MemoryManager memoryManager,
      Serializer<DnaString> dnaStringSerializer) {
    super();

    this.memoryManager = memoryManager;
    this.dnaStringSerializer = dnaStringSerializer;

    if (this.memoryManager == null) {
      throw new NullPointerException("memoryManager is null.");
    }
    if (this.dnaStringSerializer == null) {
      throw new NullPointerException("dnaStringSerializer is null.");
    }

    root = null;
  }

  @Override
  public void clear() {
    NodeVisitor<Collection<MemoryHandle>> visitor = new NodeVisitor<Collection<MemoryHandle>>() {

      final Collection<MemoryHandle> handles = new LinkedList<>();

      @Override
      public void visit(Node node, int depth) {
        if (node.isInternalNode()) {
          InternalNode internalNode = (InternalNode) node;
          handles.add(internalNode.aNodeHandle);
          handles.add(internalNode.cNodeHandle);
          handles.add(internalNode.gNodeHandle);
          handles.add(internalNode.tNodeHandle);
          handles.add(internalNode.$NodeHandle);
        } else if (node.isLeafNode()) {
          handles.add(((LeafNode) node).dnaStringHandle);
        }
      }

      @Override
      public Collection<MemoryHandle> result() {
        return handles;
      }
    };

    getRoot().traverse(visitor, 0);
    for (MemoryHandle handle : visitor.result()) {
      if (handle != null) {
        memoryManager.free(handle);
      }
    }
    super.clear();
  }

  /**
   * Returns this DNA tree's root node.
   * 
   * @return This DNA tree's root node.
   */
  protected Node getRoot() {
    return loadNode(root);
  }

  /**
   * Sets this DNA tree's root node.
   * 
   * @param root
   *          The node to set as this DNA tree's root node.
   */
  protected void setRoot(Node root) {
    assert root != null : "root is null.";

    this.root = saveNodeForReplace(root, this.root);
  }

  /**
   * Returns the {@link Node} referenced by the specified {@link MemoryHandle}.
   * 
   * @param handle
   *          The handle referencing the node to be loaded.
   * @return The node referenced by the specified handle.
   */
  private Node loadNode(MemoryHandle handle) {
    if (handle != null) {
      byte[] bytes = memoryManager.dereference(handle);
      try {
        return NODE_SERIALIZER.deserialize(bytes);
      } catch (Serializer.InvalidBytesException e) {
        System.err
            .println("Invalid byte array for a tree node is attempting to be deserialized.");
        return FLYWEIGHT;
      }
    } else {
      return FLYWEIGHT;
    }
  }

  /**
   * Returns a {@link MemoryHandle} referencing the specified {@link Node}, and
   * frees the memory referenced by the specified handle.
   * 
   * @param node
   *          The node for which a handle referencing it is to be returned.
   * @param oldHandle
   *          The handle whose referenced memory is to be freed.
   * @return A handle referencing the specified node.
   */
  private MemoryHandle saveNodeForReplace(Node node, MemoryHandle oldHandle) {
    assert node != null : "node is null.";

    if (oldHandle != null) {
      memoryManager.free(oldHandle);
    }
    return saveNode(node);
  }

  /**
   * Returns a {@link MemoryHandle} referencing the specified {@link Node}.
   * 
   * @param node
   *          The node for which a handle referencing it is to be returned.
   * @return A handle referencing the specified node.
   */
  private MemoryHandle saveNode(Node node) {
    assert node != null;

    if (node.isInternalNode() || node.isLeafNode()) {
      byte[] bytes = NODE_SERIALIZER.serialize(node);
      return memoryManager.allocate(bytes);
    } else {
      return null;
    }
  }

  @Override
  protected InternalNode createInternalNode() {
    return new InternalNode();
  }

  @Override
  protected LeafNode createLeafNode(DnaString dnaString) {
    return new LeafNode(dnaString);
  }

  /**
   * Represents a leaf node in this tree.
   */
  private class InternalNode extends DnaTreeSet.InternalNode {

    // Child node handles
    MemoryHandle aNodeHandle, cNodeHandle, gNodeHandle, tNodeHandle,
        $NodeHandle;

    /**
     * Constructs a new {@code InternalNode} with all child nodes as
     * {@link FlyweightNode}s.
     */
    InternalNode() {
      super();
    }

    /**
     * Constructs a new {@code InternalNode} with the specified child node
     * {@link MemoryHandle}s.
     * 
     * @param aNodeHandle
     *          A handle to this node's A child node.
     * @param cNodeHandle
     *          A handle to this node's C child node.
     * @param gNodeHandle
     *          A handle to this node's G child node.
     * @param tNodeHandle
     *          A handle to this node's T child node.
     * @param $NodeHandle
     *          A handle to this node's $ child node.
     */
    InternalNode(MemoryHandle aNodeHandle, MemoryHandle cNodeHandle,
        MemoryHandle gNodeHandle, MemoryHandle tNodeHandle,
        MemoryHandle $NodeHandle) {
      this.aNodeHandle = aNodeHandle;
      this.cNodeHandle = cNodeHandle;
      this.gNodeHandle = gNodeHandle;
      this.tNodeHandle = tNodeHandle;
      this.$NodeHandle = $NodeHandle;
    }

    @Override
    Node getANode() {
      return loadNode(aNodeHandle);
    }

    @Override
    void setANode(Node aNode) {
      assert aNode != null : "aNode is null.";

      aNodeHandle = saveNodeForReplace(aNode, aNodeHandle);
    }

    @Override
    Node getCNode() {
      return loadNode(cNodeHandle);
    }

    @Override
    void setCNode(Node cNode) {
      assert cNode != null : "cNode is null.";

      cNodeHandle = saveNodeForReplace(cNode, cNodeHandle);
    }

    @Override
    Node getGNode() {
      return loadNode(gNodeHandle);
    }

    @Override
    void setGNode(Node gNode) {
      assert gNode != null : "gNode is null.";

      gNodeHandle = saveNodeForReplace(gNode, gNodeHandle);
    }

    @Override
    Node getTNode() {
      return loadNode(tNodeHandle);
    }

    @Override
    void setTNode(Node tNode) {
      assert tNode != null : "tNode is null.";

      tNodeHandle = saveNodeForReplace(tNode, tNodeHandle);
    }

    @Override
    Node get$Node() {
      return loadNode($NodeHandle);
    }

    @Override
    void set$Node(Node $Node) {
      assert $Node != null : "$Node is null.";

      $NodeHandle = saveNodeForReplace($Node, $NodeHandle);
    }
  }

  /**
   * Represents a leaf node in this tree.
   */
  private class LeafNode extends DnaTreeSet.LeafNode {

    // Handle to the DNA string stored in this node (or null if no memory was
    // available for allocation)
    MemoryHandle dnaStringHandle;

    /**
     * Constructs a new {@code LeafNode} with the specified {@link DnaString} to
     * be stored in this node.
     * 
     * @param dnaString
     *          The DNA string to be stored in this node.
     */
    LeafNode(DnaString dnaString) {
      super(dnaString);
    }

    /**
     * Constructs a new {@code LeafNode} with the specified {@link DnaString}
     * {@link MemoryHandle}.
     * 
     * @param dnaStringHandle
     *          A handle to this node's DNA string.
     */
    LeafNode(MemoryHandle dnaStringHandle) {
      super(null);

      assert dnaStringHandle != null : "dnaStringHandle is null.";

      this.dnaStringHandle = dnaStringHandle;
    }

    @Override
    Node add(DnaString dnaString, int depth) throws DuplicateElementException {
      assert dnaString != null : "dnaString is null.";
      assert depth >= 0 : "depth must be a non-negative integer.";

      DnaString nodeDnaString = getDnaString();
      if (nodeDnaString == null) {
        // Replace invalid DNA string
        setDnaString(dnaString);
        return this;
      }
      if (dnaString.equals(nodeDnaString)) {
        throw new DuplicateElementException("DNA string " + dnaString
            + " already contained in this set.");
      }

      memoryManager.free(dnaStringHandle);
      InternalNode newNode = createInternalNode();
      newNode.add(nodeDnaString, depth);
      newNode.add(dnaString, depth);
      return newNode;
    }

    @Override
    boolean containsPrefix(DnaString prefix, int depth) {
      assert prefix != null : "prefix is null.";
      assert depth >= 0 : "depth must be a non-negative integer.";

      DnaString nodeDnaString = getDnaString();
      if (nodeDnaString == null) {
        // Fail gracefully if deserialization failed
        return false;
      }
      return nodeDnaString.getDnaString().startsWith(prefix.getDnaString());
    }

    @Override
    int collect(Collection<DnaString> collected) {
      assert collected != null : "collected is null.";

      DnaString nodeDnaString = getDnaString();
      if (nodeDnaString == null) {
        // Fail gracefully if deserialization failed
        return 1;
      }
      collected.add(nodeDnaString);
      return 1;
    }

    @Override
    Node remove(DnaString dnaString, int depth) throws AbsentElementException {
      Node newRoot = super.remove(dnaString, depth);
      memoryManager.free(dnaStringHandle);
      return newRoot;
    }

    @Override
    DnaString getDnaString() {
      if (dnaStringHandle == null) {
        // No memory was available to add DNA string
        return null;
      }
      byte[] bytes = memoryManager.dereference(dnaStringHandle);
      try {
        return dnaStringSerializer.deserialize(bytes);
      } catch (Serializer.InvalidBytesException e) {
        System.err.println(e.getMessage());
        return null;
      }
    }

    @Override
    void setDnaString(DnaString dnaString) {
      if (dnaString == null) {
        // Node is being created via deserialization
        return;
      }

      byte[] bytes = dnaStringSerializer.serialize(dnaString);
      dnaStringHandle = memoryManager.allocate(bytes);
      if (dnaStringHandle == null) {
        System.err.println("No memory available to add DNA string: "
            + dnaString);
      }
    }
  }

  /**
   * A serializer for instances of {@link Node}.
   */
  private class NodeSerializer implements Serializer<Node> {

    // Starting index of type flag byte
    static final int TYPE_FLAG_BYTE_INDEX = 0;

    // Starting index of node bytes
    static final int NODE_BYTES_INDEX = TYPE_FLAG_BYTE_INDEX + 1;

    // Flag byte for internal nodes
    static final byte INTERNAL_NODE_FLAG_BYTE = 0;

    // Flag byte for leaf nodes
    static final byte LEAF_NODE_FLAG_BYTE = 1;

    // Serializer for serializing and deserializing internal nodes
    final Serializer<InternalNode> INTERNAL_NODE_SERIALIZER = new InternalNodeSerializer();

    // Serializer for serializing and deserializing leaf nodes
    final Serializer<LeafNode> LEAF_NODE_SERIALIZER = new LeafNodeSerializer();

    /**
     * Constructs a new {@code NodeSerializer}.
     */
    NodeSerializer() {
      // default instantiation
    }

    @Override
    public byte[] serialize(Node node) {
      assert node != null : "node is null.";

      byte typeFlagByte;
      byte[] nodeBytes;
      if (node.isInternalNode()) {
        typeFlagByte = INTERNAL_NODE_FLAG_BYTE;
        nodeBytes = INTERNAL_NODE_SERIALIZER.serialize((InternalNode) node);
      } else if (node.isLeafNode()) {
        typeFlagByte = LEAF_NODE_FLAG_BYTE;
        nodeBytes = LEAF_NODE_SERIALIZER.serialize((LeafNode) node);
      } else {
        throw new IllegalArgumentException(
            "node is not a valid serializable node.");
      }

      byte[] bytes = new byte[NODE_BYTES_INDEX + nodeBytes.length];
      bytes[TYPE_FLAG_BYTE_INDEX] = typeFlagByte;
      System.arraycopy(nodeBytes, 0, bytes, NODE_BYTES_INDEX, nodeBytes.length);
      return bytes;
    }

    @Override
    public Node deserialize(byte[] bytes) throws InvalidBytesException {
      if (bytes == null) {
        throw new NullPointerException("bytes is null.");
      }

      // Copy bytes to protect from synchronous mutations
      byte[] copyBytes = Arrays.copyOf(bytes, bytes.length);

      byte typeFlagByte = copyBytes[TYPE_FLAG_BYTE_INDEX];
      byte[] nodeBytes = new byte[copyBytes.length - NODE_BYTES_INDEX];
      System.arraycopy(copyBytes, NODE_BYTES_INDEX, nodeBytes, 0,
          nodeBytes.length);
      switch (typeFlagByte) {
        case INTERNAL_NODE_FLAG_BYTE:
          return INTERNAL_NODE_SERIALIZER.deserialize(nodeBytes);
        case LEAF_NODE_FLAG_BYTE:
          return LEAF_NODE_SERIALIZER.deserialize(nodeBytes);
        default:
          throw new InvalidBytesException("Node type flag is invalid.");
      }
    }
  }

  /**
   * A serializer for instances of {@link InternalNode}.
   */
  private class InternalNodeSerializer implements Serializer<InternalNode> {

    // Starting index of A child node handle
    static final int A_NODE_HANDLE_BYTES_INDEX = 0;

    // Starting index of C child node handle
    final int C_NODE_HANDLE_BYTES_INDEX = A_NODE_HANDLE_BYTES_INDEX
        + Utilities.INT_BYTE_SIZE;

    // Starting index of G child node handle
    final int G_NODE_HANDLE_BYTES_INDEX = C_NODE_HANDLE_BYTES_INDEX
        + Utilities.INT_BYTE_SIZE;

    // Starting index of T child node handle
    final int T_NODE_HANDLE_BYTES_INDEX = G_NODE_HANDLE_BYTES_INDEX
        + Utilities.INT_BYTE_SIZE;

    // Starting index of $ child node handle
    final int $_NODE_HANDLE_BYTES_INDEX = T_NODE_HANDLE_BYTES_INDEX
        + Utilities.INT_BYTE_SIZE;

    // Size of serialized form
    final int SIZE = $_NODE_HANDLE_BYTES_INDEX + Utilities.INT_BYTE_SIZE;

    // Bytes for a flyweight handle
    final byte[] FLYWEIGHT_HANDLE_BYTES = Utilities.intToByteArray(-1);

    @Override
    public byte[] serialize(InternalNode node) {
      if (node == null) {
        throw new NullPointerException("node is null.");
      }

      byte[] aNodeHandleBytes = serializeHandle(node.aNodeHandle);
      byte[] cNodeHandleBytes = serializeHandle(node.cNodeHandle);
      byte[] gNodeHandleBytes = serializeHandle(node.gNodeHandle);
      byte[] tNodeHandleBytes = serializeHandle(node.tNodeHandle);
      byte[] $NodeHandleBytes = serializeHandle(node.$NodeHandle);

      int bytesLength = aNodeHandleBytes.length + cNodeHandleBytes.length
          + gNodeHandleBytes.length + tNodeHandleBytes.length
          + $NodeHandleBytes.length;
      byte[] bytes = new byte[bytesLength];
      System.arraycopy(aNodeHandleBytes, 0, bytes, A_NODE_HANDLE_BYTES_INDEX,
          aNodeHandleBytes.length);
      System.arraycopy(cNodeHandleBytes, 0, bytes, C_NODE_HANDLE_BYTES_INDEX,
          cNodeHandleBytes.length);
      System.arraycopy(gNodeHandleBytes, 0, bytes, G_NODE_HANDLE_BYTES_INDEX,
          gNodeHandleBytes.length);
      System.arraycopy(tNodeHandleBytes, 0, bytes, T_NODE_HANDLE_BYTES_INDEX,
          tNodeHandleBytes.length);
      System.arraycopy($NodeHandleBytes, 0, bytes, $_NODE_HANDLE_BYTES_INDEX,
          $NodeHandleBytes.length);
      return bytes;
    }

    /**
     * Serializes the specified {@link MemoryHanlde} into a byte array. Handles
     * the case where the specified handle references a {@link FlyweightNode}.
     * 
     * @param handle
     *          The handle to serialize.
     * @return A byte array representing the handle.
     */
    byte[] serializeHandle(MemoryHandle handle) {
      return handle != null ? Utilities.intToByteArray(handle.getResource())
          : FLYWEIGHT_HANDLE_BYTES;
    }

    @Override
    public InternalNode deserialize(byte[] bytes)
        throws Serializer.InvalidBytesException {
      if (bytes == null) {
        throw new NullPointerException("bytes is null.");
      }

      // Copy bytes to protect from synchronous mutations
      byte[] nodeBytes = Arrays.copyOf(bytes, bytes.length);

      if (nodeBytes.length != SIZE) {
        throw new InvalidBytesException(
            "Length of bytes not equal to length for a valid seralized InternalNode.");
      }

      byte[] aNodeHandleBytes = Arrays.copyOfRange(nodeBytes,
          A_NODE_HANDLE_BYTES_INDEX, A_NODE_HANDLE_BYTES_INDEX
              + Utilities.INT_BYTE_SIZE);
      byte[] cNodeHandleBytes = Arrays.copyOfRange(nodeBytes,
          C_NODE_HANDLE_BYTES_INDEX, C_NODE_HANDLE_BYTES_INDEX
              + Utilities.INT_BYTE_SIZE);
      byte[] gNodeHandleBytes = Arrays.copyOfRange(nodeBytes,
          G_NODE_HANDLE_BYTES_INDEX, G_NODE_HANDLE_BYTES_INDEX
              + Utilities.INT_BYTE_SIZE);
      byte[] tNodeHandleBytes = Arrays.copyOfRange(nodeBytes,
          T_NODE_HANDLE_BYTES_INDEX, T_NODE_HANDLE_BYTES_INDEX
              + Utilities.INT_BYTE_SIZE);
      byte[] $NodeHandleBytes = Arrays.copyOfRange(nodeBytes,
          $_NODE_HANDLE_BYTES_INDEX, $_NODE_HANDLE_BYTES_INDEX
              + Utilities.INT_BYTE_SIZE);

      MemoryHandle aNodeHandle = deserializeHandle(aNodeHandleBytes);
      MemoryHandle cNodeHandle = deserializeHandle(cNodeHandleBytes);
      MemoryHandle gNodeHandle = deserializeHandle(gNodeHandleBytes);
      MemoryHandle tNodeHandle = deserializeHandle(tNodeHandleBytes);
      MemoryHandle $NodeHandle = deserializeHandle($NodeHandleBytes);

      return new InternalNode(aNodeHandle, cNodeHandle, gNodeHandle,
          tNodeHandle, $NodeHandle);
    }

    /**
     * Returns a {@link MemoryHandle} represented by the specified byte array.
     * Handles the case where the specified bytes represent a
     * {@link FlyweightNode}.
     * 
     * @param bytes
     *          The byte array representing the handle.
     * @return A handle represented by the specified byte array.
     */
    MemoryHandle deserializeHandle(byte[] bytes) {
      return Arrays.equals(bytes, FLYWEIGHT_HANDLE_BYTES) ? null
          : new MemoryHandle(Utilities.byteArrayToInt(bytes));
    }
  }

  /**
   * A serializer for instances of {@link LeafNode}.
   */
  private class LeafNodeSerializer implements Serializer<LeafNode> {

    // Size of serialized form
    final int SIZE = Utilities.INT_BYTE_SIZE;

    @Override
    public byte[] serialize(LeafNode node) {
      if (node == null) {
        throw new NullPointerException("node is null.");
      }

      return Utilities.intToByteArray(node.dnaStringHandle.getResource());
    }

    @Override
    public LeafNode deserialize(byte[] bytes)
        throws Serializer.InvalidBytesException {
      if (bytes == null) {
        throw new NullPointerException("bytes is null.");
      }

      // Copy bytes to protect from synchronous mutations
      byte[] nodeBytes = Arrays.copyOf(bytes, bytes.length);

      if (nodeBytes.length != SIZE) {
        throw new InvalidBytesException(
            "Length of bytes not equal to length for a valid seralized LeafNode.");
      }

      MemoryHandle dnaStringHandle = new MemoryHandle(
          Utilities.byteArrayToInt(nodeBytes));
      return new LeafNode(dnaStringHandle);
    }
  }
}
