import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintStream;
import java.io.PrintWriter;

/**
 * A DNA sequence management package for storing and searching DNA string
 * records.
 * <p>
 * Validates the command line arguments and manages the initialization of the
 * various system components. Hands off execution to a command processor that
 * manages the retrieving of commands from the command file and makes the
 * necessary calls to the system components in order to carry out those
 * commands.
 * 
 * @author Michael D. Naper, Jr. <MichaelNaper.com>
 * @version 2013.01.17
 */
public class DnaDatabase {

  // Filename of data file
  private static final String DATA_FILENAME = "p4bin.dat";

  // "Standard" (i.e., default) output stream
  private static final PrintStream STANDARD_OUT = System.out;

  /**
   * Main method for program. Validates arguments and begins program execution.
   * 
   * @param args
   *          The program arguments of the form
   *          "<command-file> <num-buffers> <buffer-size> [<output-file>]".
   */
  public static void main(String[] args) {
    if (args.length < 3) {
      System.err.println("Command arguments must take the form: "
          + "\"<command-file> <num-buffers> <buffer-size> "
          + "[<output-file>]\"");
      return;
    }

    String commandFilename = args[0];

    int numBuffers;
    try {
      numBuffers = Integer.parseInt(args[1]);
    } catch (NumberFormatException e) {
      System.err.println("Must enter an integer as second argument.\n"
          + "Command arguments must take the form: "
          + "\"<command-file> <num-buffers> <buffer-size> "
          + "[<output-file>]\"");
      return;
    }
    if (numBuffers < 1) {
      System.err.println("Must enter a positive number of buffers.\n"
          + "Command arguments must take the form: "
          + "\"<command-file> <num-buffers> <buffer-size> "
          + "[<output-file>]\"");
      return;
    }

    int bufferSize;
    try {
      bufferSize = Integer.parseInt(args[2]);
    } catch (NumberFormatException e) {
      System.err.println("Must enter an integer as third argument.\n"
          + "Command arguments must take the form: "
          + "\"<command-file> <num-buffers> <buffer-size> "
          + "[<output-file>]\"");
      return;
    }
    if (bufferSize < 1) {
      System.err.println("Must enter a positive number for buffer size.\n"
          + "Command arguments must take the form: "
          + "\"<command-file> <num-buffers> <buffer-size> "
          + "[<output-file>]\"");
      return;
    }

    String outputFilename = null;
    if (args.length > 3) {
      outputFilename = args[3];
    }

    initializeComponentsAndExecute(commandFilename, numBuffers, bufferSize,
        outputFilename);
  }

  /**
   * Initializes system components and begins program execution.
   * 
   * @param commandFilename
   *          The filename of the file containing the commands to execute.
   * @param numBuffers
   *          The number of buffers to allocate for the buffer pool before a
   *          buffer must be replaced.
   * @param bufferSize
   *          The maximum number of bytes each buffer can hold. The data file is
   *          split into blocks of this size.
   * @param outputFilename
   *          The filename of the file to which to print output, or {@code null}
   *          if output is to print to "standard" output.
   */
  private static final void initializeComponentsAndExecute(
      String commandFilename, int numBuffers, int bufferSize,
      String outputFilename) {
    assert commandFilename != null : "commandFilename is null.";
    assert numBuffers > 0 : "numBuffers must be a positive integer.";
    assert bufferSize > 0 : "bufferSize must be a positive integer.";

    File dataFile = new File(DATA_FILENAME);
    BufferPool bufferPool;
    try {
      bufferPool = new BufferPool(dataFile, numBuffers, bufferSize);
    } catch (FileNotFoundException e) {
      System.err.println(DATA_FILENAME + " cannot be found.\n");
      return;
    }
    MemoryPool memoryPool = new MemoryPool(bufferPool, 0);
    FreeList freeList = new FreeList();
    MemoryManager memoryManager = new MemoryManager(memoryPool, freeList,
        bufferSize);
    Serializer<DnaString> dnaStringSerializer = DnaStringSerializer
        .getInstance();
    DnaTreeSet dnaTreeSet = new DiskDnaTreeSet(memoryManager,
        dnaStringSerializer);
    RecordDatabase recordDatabase = new RecordDatabase(dnaTreeSet);
    File commandFile = new File(commandFilename);
    CommandParser commandParser;
    try {
      commandParser = new CommandParser(commandFile);
    } catch (FileNotFoundException e) {
      System.err.println(commandFilename + " cannot be found.\n"
          + "Command arguments must take the form: "
          + "\"<command-file> <num-buffers> <buffer-size> "
          + "[<output-file>]\"");
      return;
    }
    PrintWriter printWriter = createPrintWriter(outputFilename);
    Controller controller = new Controller(recordDatabase, memoryManager,
        bufferPool, commandParser, printWriter);

    controller.execute();
    printWriter.flush();

    commandParser.close();
    printWriter.close();
  }

  /**
   * Creates a print writer whose destination is the file referred to by the
   * specified filename, or is "standard" output if the filename is {@code null}
   * or the file cannot be found.
   * 
   * @param outputFilename
   *          The name of the file to be used as the destination of the print
   *          writer.
   * @return A print writer with the appropriate output stream.
   */
  private static final PrintWriter createPrintWriter(String outputFilename) {
    if (outputFilename == null) {
      return new PrintWriter(STANDARD_OUT);
    }
    try {
      return new PrintWriter(outputFilename);
    } catch (FileNotFoundException e) {
      return new PrintWriter(STANDARD_OUT);
    }
  }
}
