import java.util.HashSet;
import java.util.Set;

/**
 * A DNA string, defined on the alphabet A, C, G, and T.
 * 
 * @author Michael D. Naper, Jr. <MichaelNaper.com>
 * @version 2013.01.17
 */
public class DnaString {

  /**
   * Alphabet on which DNA strings are defined.
   */
  public static final char[] ALPHABET = { 'A', 'C', 'G', 'T' };

  // Set of valid characters based on the DNA alphabet
  private static final Set<Character> VALID_CHARS = buildValidCharsSet();

  // DNA string
  private final String dnaString;

  /**
   * Builds the valid character set based on the DNA alphabet.
   * 
   * @return A set of valid DNA characters.
   */
  private static Set<Character> buildValidCharsSet() {
    Set<Character> validChars = new HashSet<>();
    for (char c : ALPHABET) {
      validChars.add(c);
    }
    return validChars;
  }

  /**
   * Constructs a new {@code DnaString} from the specified DNA string.
   * 
   * @param dnaString
   *          The DNA string.
   */
  public DnaString(String dnaString) {
    this.dnaString = dnaString;

    if (this.dnaString == null) {
      throw new NullPointerException("dnaString is null.");
    }
    checkValidDnaString(this.dnaString);
  }

  /**
   * Checks if the specified string is a valid DNA string.
   * 
   * @param string
   *          The string to check.
   * @throws IllegalArgumentException
   *           If the specified string is not a valid DNA string.
   */
  private void checkValidDnaString(String string) {
    assert string != null : "string is null.";

    for (char c : string.toCharArray()) {
      if (!VALID_CHARS.contains(c)) {
        throw new IllegalArgumentException(
            "String contains illegal character: " + c);
      }
    }
  }

  /**
   * Returns the underlying DNA string.
   * 
   * @return The underlying DNA string.
   */
  public String getDnaString() {
    return dnaString;
  }

  @Override
  public String toString() {
    return dnaString;
  }

  @Override
  public int hashCode() {
    return dnaString.hashCode();
  }

  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (!(obj instanceof DnaString)) {
      return false;
    }
    return this.dnaString.equals(((DnaString) obj).getDnaString());
  }
}
