import java.util.HashSet;
import java.util.Set;

/**
 * A DNA string descriptor, defined on the DNA alphabet A, C, G, and T as well
 * as special characters.
 * 
 * @author Michael D. Naper, Jr. <MichaelNaper.com>
 * @version 2013.01.17
 */
public class DnaStringDescriptor {

  /**
   * Termination character for DNA string descriptors.
   */
  public static final char TERMINATION_CHAR = '$';

  /**
   * Alphabet of special characters on which DNA string descriptors are defined.
   */
  public static final char[] SPECIAL_CHAR_ALPHABET = { TERMINATION_CHAR };

  // Set of valid characters based on the DNA descriptor alphabet
  private static final Set<Character> VALID_CHARS = buildValidCharsSet();

  // DNA string descriptor
  private final String dnaStringDescriptor;

  /**
   * Builds the valid character set based on the DNA descriptor alphabet.
   * 
   * @return A set of valid DNA descriptor characters.
   */
  private static Set<Character> buildValidCharsSet() {
    Set<Character> validChars = new HashSet<>();
    for (char c : DnaString.ALPHABET) {
      validChars.add(c);
    }
    for (char c : SPECIAL_CHAR_ALPHABET) {
      validChars.add(c);
    }
    return validChars;
  }

  /**
   * Constructs a new {@code DnaStringDescriptor} from the specified DNA string
   * descriptor.
   * 
   * @param dnaStringDescriptor
   *          The DNA string descriptor.
   */
  public DnaStringDescriptor(String dnaStringDescriptor) {
    this.dnaStringDescriptor = dnaStringDescriptor;

    if (this.dnaStringDescriptor == null) {
      throw new NullPointerException("dnaStringDescriptor is null.");
    }
    checkValidDnaStringDescriptor(this.dnaStringDescriptor);
  }

  /**
   * Checks if the specified string is a valid DNA string descriptor.
   * 
   * @param string
   *          The string to check.
   * @throws IllegalArgumentException
   *           If the specified string is not a valid DNA string descriptor.
   */
  private void checkValidDnaStringDescriptor(String string) {
    assert string != null : "string is null.";

    for (char c : string.toCharArray()) {
      if (!VALID_CHARS.contains(c)) {
        throw new IllegalArgumentException(
            "String contains illegal character: " + c);
      }
    }
    int terminationCharIndex = string.indexOf(TERMINATION_CHAR);
    if (terminationCharIndex != -1
        && terminationCharIndex < string.length() - 1) {
      throw new IllegalArgumentException(
          "String contains termination character " + TERMINATION_CHAR
              + " at illegal position.");
    }
  }

  /**
   * Constructs a new {@code DnaStringDescriptor} from the specified
   * {@link DnaString}.
   * 
   * @param dnaString
   *          The DNA string.
   */
  public DnaStringDescriptor(DnaString dnaString) {
    if (dnaString == null) {
      throw new NullPointerException("dnaString is null.");
    }

    // DNA strings are always valid DNA string descriptors
    dnaStringDescriptor = dnaString.getDnaString();
  }

  /**
   * Returns the underlying DNA string descriptor.
   * 
   * @return The underlying DNA string descriptor.
   */
  public String getDnaStringDescriptor() {
    return dnaStringDescriptor;
  }

  /**
   * Returns {@code true} if the specified {@link DnaString} matches this DNA
   * string descriptor.
   * 
   * @param dnaString
   *          The DNA string against which to match.
   * @return {@code true} if the specified DNA string matches this DNA string
   *         descriptor.
   */
  public boolean matches(DnaString dnaString) {
    if (dnaString == null) {
      return false;
    }
    if (dnaStringDescriptor.endsWith(String.valueOf(TERMINATION_CHAR))) {
      return dnaString.getDnaString().equals(
          dnaStringDescriptor.substring(0, dnaStringDescriptor.length() - 1));
    }
    if (dnaString.getDnaString().startsWith(dnaStringDescriptor)) {
      return true;
    }
    return false;
  }

  /**
   * Returns a copy of this {@code DnaStringDescriptor} with all special
   * characters removed.
   * 
   * @return A copy of this {@code DnaStringDescriptor} with all special
   *         characters removed.
   */
  public DnaStringDescriptor removeSpecialChars() {
    String removedSpecialChars = dnaStringDescriptor;
    for (char specialChar : SPECIAL_CHAR_ALPHABET) {
      removedSpecialChars = removedSpecialChars.replaceAll("\\" + specialChar,
          "");
    }
    return new DnaStringDescriptor(removedSpecialChars);
  }

  @Override
  public String toString() {
    return dnaStringDescriptor;
  }

  @Override
  public int hashCode() {
    return dnaStringDescriptor.hashCode();
  }

  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (!(obj instanceof DnaStringDescriptor)) {
      return false;
    }
    return this.dnaStringDescriptor.equals(((DnaStringDescriptor) obj)
        .getDnaStringDescriptor());
  }
}
