import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

/**
 * A serializer for instances of {@link DnaString}.
 * 
 * @author Michael D. Naper, Jr. <MichaelNaper.com>
 * @version 2013.01.07
 */
public class DnaStringSerializer implements Serializer<DnaString> {

  // Singleton instance for the lifetime of the program
  private static final DnaStringSerializer INSTANCE = new DnaStringSerializer();

  // Number of bytes used to store the DNA string length
  private static final int LENGTH_SIZE = 2;

  /**
   * Maximum allowable number of characters for serialization.
   */
  public static final int MAX_LENGTH = (1 << (LENGTH_SIZE * Byte.SIZE)) - 1;

  // Length of binary codes
  private static final int BINARY_CODE_LENGTH = 2;

  // Mapping from DNA character to binary code
  private static final Map<Character, Byte> CHAR_TO_BINARY_MAP = buildCharToBinaryMap();

  // Mapping from binary code to DNA character
  private static final Map<Byte, Character> BINARY_TO_CHAR_MAP = buildBinaryToCharMap();

  /**
   * Builds a mapping from DNA characters to their corresponding binary codes.
   * 
   * @return A mapping from DNA characters to their corresponding binary codes.
   */
  private static Map<Character, Byte> buildCharToBinaryMap() {
    Map<Character, Byte> charToBinaryMap = new HashMap<>();
    charToBinaryMap.put('A', (byte) 0b00);
    charToBinaryMap.put('C', (byte) 0b01);
    charToBinaryMap.put('G', (byte) 0b10);
    charToBinaryMap.put('T', (byte) 0b11);
    return charToBinaryMap;
  }

  /**
   * Builds a mapping from binary codes to their corresponding DNA characters.
   * 
   * @return A mapping from binary codes to their corresponding DNA characters.
   */
  private static Map<Byte, Character> buildBinaryToCharMap() {
    Map<Byte, Character> binaryToCharMap = new HashMap<>();
    for (char c : CHAR_TO_BINARY_MAP.keySet()) {
      binaryToCharMap.put(CHAR_TO_BINARY_MAP.get(c), c);
    }
    return binaryToCharMap;
  }

  private DnaStringSerializer() {
    // disable instantiation externally
  }

  /**
   * Returns a singleton instance of {@code DnaStringSerializer}.
   * 
   * @return A singleton instance of {@code DnaStringSerializer}.
   */
  public static DnaStringSerializer getInstance() {
    return INSTANCE;
  }

  @Override
  public byte[] serialize(DnaString dnaString) {
    if (dnaString == null) {
      throw new NullPointerException("dnaString is null.");
    }
    if (dnaString.getDnaString().length() > MAX_LENGTH) {
      throw new IllegalArgumentException(
          "dnaString cannot be serialized because it exceeds the maximum allowable length.");
    }

    // create byte array
    int bytesLength = LENGTH_SIZE
        + (int) Math.ceil((double) (BINARY_CODE_LENGTH * dnaString
            .getDnaString().length()) / Byte.SIZE);
    byte[] bytes = new byte[bytesLength];

    // encode and store DNA string length
    byte[] lengthBytes = Utilities.intToByteArray(dnaString.getDnaString()
        .length());
    int numLengthBytes = Math.min(LENGTH_SIZE, lengthBytes.length);
    System.arraycopy(lengthBytes, lengthBytes.length - numLengthBytes, bytes,
        0, numLengthBytes);

    // encode and store DNA string
    char[] dnaChars = dnaString.getDnaString().toCharArray();
    for (int i = 0; i < dnaChars.length; i++) {
      byte binaryCode = CHAR_TO_BINARY_MAP.get(dnaChars[i]);
      int bytesIndex = LENGTH_SIZE + (BINARY_CODE_LENGTH * i) / Byte.SIZE;
      int shiftAmt = Byte.SIZE - (i % (Byte.SIZE / BINARY_CODE_LENGTH) + 1)
          * BINARY_CODE_LENGTH;
      bytes[bytesIndex] |= binaryCode << shiftAmt;
    }

    return bytes;
  }

  @Override
  public DnaString deserialize(byte[] bytes) throws InvalidBytesException {
    if (bytes == null) {
      throw new NullPointerException("bytes is null.");
    }

    // Copy bytes to protect from synchronous mutations
    byte[] dnaStringBytes = Arrays.copyOf(bytes, bytes.length);

    if (dnaStringBytes.length < LENGTH_SIZE) {
      throw new InvalidBytesException(
          "Length of bytes less than minimum length for a valid seralized DnaString.");
    }

    // extract DNA string length
    byte[] lengthBytes = new byte[Utilities.INT_BYTE_SIZE];
    int numLengthBytes = Math.min(LENGTH_SIZE, lengthBytes.length);
    System.arraycopy(dnaStringBytes, 0, lengthBytes, lengthBytes.length
        - numLengthBytes, numLengthBytes);
    int length = Utilities.byteArrayToInt(lengthBytes);

    // extract DNA string
    int numBytes = (int) Math.ceil((double) (BINARY_CODE_LENGTH * length)
        / Byte.SIZE);
    int numCharsPerByte = Byte.SIZE / BINARY_CODE_LENGTH;
    int binaryCodeMask = (1 << BINARY_CODE_LENGTH) - 1;
    StringBuilder dnaStringBuilder = new StringBuilder();
    byteLoop: for (int i = 0; i < numBytes; i++) {
      for (int j = 0; j < numCharsPerByte; j++) {
        if (dnaStringBuilder.length() >= length) {
          break byteLoop;
        }
        int shiftAmt = Byte.SIZE - (j + 1) * BINARY_CODE_LENGTH;
        byte binaryCode = (byte) ((dnaStringBytes[LENGTH_SIZE + i] >> shiftAmt) & binaryCodeMask);
        char dnaChar = BINARY_TO_CHAR_MAP.get(binaryCode);
        dnaStringBuilder.append(dnaChar);
      }
    }

    return new DnaString(dnaStringBuilder.toString());
  }
}
