import java.util.List;
import java.util.ListIterator;

/**
 * A list to keep track of free blocks in a memory pool.
 * <p>
 * Uses the circular first fit rule for selecting which free block to use for a
 * memory request.
 * 
 * @author Michael D. Naper, Jr. <MichaelNaper.com>
 * @version 2013.01.17
 */
public class FreeList {

  // List of free blocks in order of increasing address (not necessarily
  // starting at the beginning of the list)
  private final List<Block> freeList;

  // Iterator over free blocks
  private final ListIterator<Block> freeListIter;

  /**
   * Constructs a new, empty {@code FreeList}.
   */
  public FreeList() {
    freeList = new LinkedList<>();
    freeListIter = new CircularListIterator<>(freeList);
  }

  /**
   * Adds a block at the specified memory address and with the specified size to
   * this free list.
   * <p>
   * Assumes that a block with the specified memory address is not already
   * contained in this free list. Behavior of this free list otherwise is
   * undefined.
   * 
   * @param address
   *          The memory address of the block.
   * @param size
   *          The size of the block.
   */
  public void addBlock(MemoryHandle address, int size) {
    if (address == null) {
      throw new NullPointerException("address is null.");
    }
    if (size <= 0) {
      throw new IllegalArgumentException("size must be a positive integer.");
    }

    if (freeList.isEmpty()) {
      freeListIter.add(new Block(address, size));
      return;
    }

    // find position to insert block
    seekForAddress(address);

    // merge any adjacent neighbor blocks
    MemoryHandle finalAddress = address;
    int finalSize = size;
    Block neighbor = freeListIter.previous();
    if (neighbor.address.getResource() + neighbor.size == address.getResource()) {
      finalAddress = neighbor.address;
      finalSize += neighbor.size;
      freeListIter.remove();
    } else {
      freeListIter.next();
    }
    if (freeListIter.hasNext()) {
      neighbor = freeListIter.next();
      if (neighbor.address.getResource() == address.getResource() + size) {
        finalSize += neighbor.size;
        freeListIter.remove();
      } else {
        freeListIter.previous();
      }
    }

    // add block to free list
    freeListIter.add(new Block(finalAddress, finalSize));
    freeListIter.previous();
  }

  /**
   * Moves this free list iterator's cursor position to where a block with the
   * specified memory address should be located.
   * 
   * @param address
   *          The memory address to which to seek.
   */
  private void seekForAddress(MemoryHandle address) {
    assert address != null : "address is null.";

    Block curr = freeListIter.next();
    MemoryHandle lastAddress;
    if (address.getResource() < curr.address.getResource()) {
      freeListIter.previous(); // skip same block
      do {
        lastAddress = curr.address;
        curr = freeListIter.previous();
      } while (address.getResource() < curr.address.getResource()
          && curr.address.getResource() < lastAddress.getResource());
      freeListIter.next();
    } else {
      do {
        lastAddress = curr.address;
        curr = freeListIter.next();
      } while (address.getResource() > curr.address.getResource()
          && curr.address.getResource() > lastAddress.getResource());
      freeListIter.previous();
    }
  }

  /**
   * Returns the address in the associated memory pool where space with the
   * specified required size can be properly allocated, or {@code null} if space
   * is unable to be allocated.
   * <p>
   * The circular next fit rule is used for selecting which free block to use
   * for a memory request. That is, the next free block (starting from the last
   * block added or considered for allocation) that is large enough to store the
   * requested space will be used to service the request (if any such block
   * exists). If the entirety of the block is not needed, the remaining space
   * will make up a new free block and be returned to this free list.
   * 
   * @param requiredSize
   *          The required size in bytes of the space requested.
   * @return The memory pool address where space with the specified required
   *         size can be properly allocated, or {@code null} if space is unable
   *         to be allocated.
   */
  public MemoryHandle requestBlock(int requiredSize) {
    if (requiredSize <= 0) {
      throw new IllegalArgumentException(
          "requiredSize must be a positive integer.");
    }

    Block grabbedBlock = null;
    if (!freeList.isEmpty()) {
      int breakIndex = freeListIter.nextIndex();
      do {
        Block freeblock = freeListIter.next();
        if (requiredSize <= freeblock.size) {
          grabbedBlock = freeblock;
          freeListIter.remove();
        }
      } while (grabbedBlock == null && freeListIter.nextIndex() != breakIndex);
    }

    // no free block large enough to service request
    if (grabbedBlock == null) {
      return null;
    }

    // return remaining space to free block list
    if (requiredSize < grabbedBlock.size) {
      MemoryHandle address = new MemoryHandle(
          grabbedBlock.address.getResource() + requiredSize);
      int size = grabbedBlock.size - requiredSize;
      freeListIter.add(new Block(address, size));
      freeListIter.previous();
    }

    return grabbedBlock.address;
  }

  @Override
  public String toString() {
    if (freeList.isEmpty()) {
      return "[]";
    }

    // find start of monotonically increasing addresses
    ListIterator<Block> iter = new CircularListIterator<>(freeList,
        freeList.size() - 1);
    MemoryHandle currAddress = iter.next().address;
    MemoryHandle lastAddress;
    do {
      lastAddress = currAddress;
      currAddress = iter.next().address;
    } while (currAddress.getResource() > lastAddress.getResource());
    iter.previous();

    StringBuilder sb = new StringBuilder();
    sb.append("[");
    for (int i = 0; i < freeList.size() - 1; i++) {
      sb.append(iter.next());
      sb.append(", ");
    }
    sb.append(iter.next());
    sb.append("]");
    return sb.toString();
  }

  /**
   * Represents a block in a memory pool.
   */
  private static class Block {

    // Memory pool address of block
    final MemoryHandle address;

    // Size of block
    final int size;

    /**
     * Constructs a new {@code Block} located at the specified memory pool
     * address and with the specified size.
     * 
     * @param address
     *          The memory pool address of the block.
     * @param size
     *          The size of the block.
     */
    Block(MemoryHandle address, int size) {
      assert address != null : "address is null.";
      assert size > 0 : "size must be a positive integer.";

      this.address = address;
      this.size = size;
    }

    @Override
    public String toString() {
      return "[" + address.getResource() + ", "
          + (address.getResource() + size - 1) + "] (" + size + " bytes)";
    }
  }
}
