import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.NoSuchElementException;

/**
 * Doubly-linked list implementation of the {@code List} interface. Implements
 * all optional list operations, and permits all elements (including
 * {@code null}).
 * <p>
 * All of the operations perform as could be expected for a doubly-linked list.
 * Operations that index into the list will traverse the list from the beginning
 * or the end, whichever is closer to the specified index.
 * <p>
 * Note that this implementation is not synchronized. If multiple threads access
 * a linked list concurrently, and at least one of the threads modifies the list
 * structurally, it must be synchronized externally.
 * 
 * @author Michael D. Naper, Jr. <MichaelNaper.com>
 * @version 2013.01.17
 * @param <E>
 *          The type of elements held in this collection.
 * @see List
 * @see ArrayList
 */
public class LinkedList<E> implements List<E> {

  // The head node of this list
  private Node<E> head;

  // The tail node of this list
  private Node<E> tail;

  // The number of elements stored in this list
  private int size;

  /**
   * Constructs a new, empty {@code LinkedList}.
   */
  public LinkedList() {
    head = tail = null;
    size = 0;
  }

  /**
   * Constructs a new {@code LinkedList} containing the elements of the
   * specified collection in the order that they are returned by the
   * collection's iterator.
   * 
   * @param collection
   *          The collection whose elements are to be added to this list.
   */
  public LinkedList(Collection<? extends E> collection) {
    this();
    addAll(collection);
  }

  @Override
  public boolean add(E element) {
    linkAtTail(element);
    return true;
  }

  @Override
  public void add(int index, E element) {
    checkPositionIndex(index);

    if (index == size) {
      linkAtTail(element);
    } else {
      linkBefore(element, getNode(index));
    }
  }

  @Override
  public boolean addAll(Collection<? extends E> collection) {
    return addAll(size, collection);
  }

  @Override
  public boolean addAll(int index, Collection<? extends E> collection) {
    checkPositionIndex(index);

    // Prevent adding during concurrent modification of collection
    Object[] elements = collection.toArray();
    int collectionSize = elements.length;
    if (collectionSize == 0) {
      return false;
    }

    Node<E> predecessor, successor;
    if (index == size) {
      successor = null;
      predecessor = tail;
    } else {
      successor = getNode(index);
      predecessor = successor.prev;
    }

    for (Object element : elements) {
      @SuppressWarnings("unchecked")
      Node<E> newNode = new Node<E>((E) element, predecessor, null);
      if (predecessor == null) {
        head = newNode;
      } else {
        predecessor.next = newNode;
      }
      predecessor = newNode;
    }

    if (successor == null) {
      tail = predecessor;
    } else {
      predecessor.next = successor;
      successor.prev = predecessor;
    }

    size += collectionSize;
    return true;
  }

  @Override
  public boolean contains(Object o) {
    return indexOf(o) != -1;
  }

  @Override
  public boolean containsAll(Collection<?> collection) {
    for (Object o : collection) {
      if (!contains(o)) {
        return false;
      }
    }
    return true;
  }

  @Override
  public int indexOf(Object o) {
    int index = 0;
    if (o == null) {
      for (Node<E> curr = head; curr != null; curr = curr.next) {
        if (curr.element == null) {
          return index;
        }
        index++;
      }
    } else {
      for (Node<E> curr = head; curr != null; curr = curr.next) {
        if (o.equals(curr.element)) {
          return index;
        }
        index++;
      }
    }
    return -1;
  }

  @Override
  public int lastIndexOf(Object o) {
    int index = size - 1;
    if (o == null) {
      for (Node<E> curr = tail; curr != null; curr = curr.prev) {
        if (curr.element == null) {
          return index;
        }
        index--;
      }
    } else {
      for (Node<E> curr = tail; curr != null; curr = curr.prev) {
        if (o.equals(curr.element)) {
          return index;
        }
        index--;
      }
    }
    return -1;
  }

  @Override
  public E get(int index) {
    checkElementIndex(index);
    return getNode(index).element;
  }

  @Override
  public E set(int index, E element) {
    checkElementIndex(index);
    final Node<E> node = getNode(index);
    final E oldElement = node.element;
    node.element = element;
    return oldElement;
  }

  @Override
  public boolean remove(Object o) {
    if (o == null) {
      for (Node<E> curr = head; curr != null; curr = curr.next) {
        if (curr.element == null) {
          unlink(curr);
          return true;
        }
      }
    } else {
      for (Node<E> curr = head; curr != null; curr = curr.next) {
        if (o.equals(curr.element)) {
          unlink(curr);
          return true;
        }
      }
    }
    return false;
  }

  @Override
  public E remove(int index) {
    checkElementIndex(index);
    return unlink(getNode(index));
  }

  @Override
  public boolean removeAll(Collection<?> collection) {
    boolean modified = false;
    for (Node<E> curr = head, next; curr != null;) {
      next = curr.next;
      if (collection.contains(curr.element)) {
        unlink(curr);
        modified = true;
      }
      curr = next;
    }
    return modified;
  }

  @Override
  public boolean retainAll(Collection<?> collection) {
    boolean modified = false;
    for (Node<E> curr = head, next; curr != null;) {
      next = curr.next;
      if (!collection.contains(curr.element)) {
        unlink(curr);
        modified = true;
      }
      curr = next;
    }
    return modified;
  }

  @Override
  public void clear() {
    // Help garbage collector
    for (Node<E> curr = head; curr != null;) {
      Node<E> next = curr.next;
      curr.element = null;
      curr.prev = null;
      curr.next = null;
      curr = next;
    }

    head = tail = null;
    size = 0;
  }

  /**
   * Links the specified element before the specified successor node.
   * 
   * @param element
   *          The element to be linked.
   * @param successor
   *          The successor node.
   */
  private void linkBefore(E element, Node<E> successor) {
    if (successor == null) {
      linkAtTail(element);
      return;
    }
    final Node<E> predecessor = successor.prev;
    if (predecessor == null) {
      linkAtHead(element);
      return;
    }
    final Node<E> newNode = new Node<>(element, predecessor, successor);
    predecessor.next = newNode;
    successor.prev = newNode;
    size++;
  }

  /**
   * Unlinks the specified node.
   * 
   * @param node
   *          The node to be unlinked.
   * @return The element contained in the node that was unlinked.
   */
  private E unlink(Node<E> node) {
    assert node != null : "node is null.";

    final Node<E> prev = node.prev;
    if (prev == null) {
      return unlinkAtHead();
    }
    final Node<E> next = node.next;
    if (next == null) {
      return unlinkAtTail();
    }
    final E element = node.element;
    prev.next = next;
    next.prev = prev;
    size--;
    // Help garbage collector
    node.element = null;
    node.prev = null;
    node.next = null;
    return element;
  }

  /**
   * Links the specified element at the head of this list.
   * 
   * @param element
   *          The element to be linked.
   */
  private void linkAtHead(E element) {
    final Node<E> oldHead = head;
    final Node<E> newNode = new Node<>(element, null, oldHead);
    head = newNode;
    if (oldHead == null) {
      tail = newNode;
    } else {
      oldHead.prev = newNode;
    }
    size++;
  }

  /**
   * Unlinks the element at the head of this list.
   * 
   * @return The element that was previously at the head of this list.
   */
  private E unlinkAtHead() {
    if (head == null) {
      return null;
    }
    final Node<E> oldHead = head;
    final E element = oldHead.element;
    final Node<E> newHead = oldHead.next;
    head = newHead;
    if (newHead == null) {
      tail = null;
    } else {
      newHead.prev = null;
    }
    size--;
    // Help garbage collector
    oldHead.element = null;
    oldHead.next = null;
    return element;
  }

  /**
   * Links the specified element at the tail of this list.
   * 
   * @param element
   *          The element to be linked.
   */
  private void linkAtTail(E element) {
    final Node<E> oldTail = tail;
    final Node<E> newNode = new Node<>(element, oldTail, null);
    tail = newNode;
    if (oldTail == null) {
      head = newNode;
    } else {
      oldTail.next = newNode;
    }
    size++;
  }

  /**
   * Unlinks the element at the tail of this list.
   * 
   * @return The element that was previously at the tail of this list.
   */
  private E unlinkAtTail() {
    if (tail == null) {
      return null;
    }
    final Node<E> oldTail = tail;
    final E element = oldTail.element;
    final Node<E> newTail = oldTail.prev;
    tail = newTail;
    if (newTail == null) {
      head = null;
    } else {
      newTail.next = null;
    }
    size--;
    // Help garbage collector
    oldTail.element = null;
    oldTail.next = null;
    return element;
  }

  @Override
  public List<E> subList(int fromIndex, int toIndex) {
    checkElementIndex(fromIndex);
    checkElementIndex(toIndex);
    if (fromIndex > toIndex) {
      throw new IllegalArgumentException("fromIndex(" + fromIndex
          + ") > toIndex(" + toIndex + ")");
    }
    List<E> subList = new LinkedList<>();
    Node<E> curr = getNode(fromIndex);
    for (int index = fromIndex; index < toIndex; index++) {
      subList.add(curr.element);
      curr = curr.next;
    }
    return subList;
  }

  /**
   * Returns the node at the specified index.
   * 
   * @param index
   *          The index of the node to be returned.
   * @return The node at the specified index.
   */
  private Node<E> getNode(int index) {
    assert isElementIndex(index) : "index is not an element index.";

    if (index < (size >> 1)) {
      Node<E> curr = head;
      for (int i = 0; i < index; i++) {
        curr = curr.next;
      }
      return curr;
    } else {
      Node<E> curr = tail;
      for (int i = size - 1; i > index; i--) {
        curr = curr.prev;
      }
      return curr;
    }
  }

  /**
   * Returns {@code true} if the specified index is the index of an existing
   * element.
   * 
   * @param index
   *          The index to be checked.
   * @return {@code true} if the specified index is an element index.
   */
  private boolean isElementIndex(int index) {
    return index >= 0 && index < size;
  }

  /**
   * Returns {@code true} if the specified index is the index of a valid
   * position for an add operation.
   * 
   * @param index
   *          The index to be checked.
   * @return {@code true} if the specified index is a valid position index.
   */
  private boolean isPositionIndex(int index) {
    return index >= 0 && index <= size;
  }

  /**
   * Checks whether the specified index is the index of an existing element.
   * 
   * @param index
   *          The index to be checked.
   * @throws IndexOutOfBoundsException
   *           If the specified index is not the index of an existing element.
   */
  private void checkElementIndex(int index) {
    if (!isElementIndex(index)) {
      throw new IndexOutOfBoundsException(outOfBoundsMsg(index));
    }
  }

  /**
   * Checks whether the specified index is the index of a valid position for an
   * add operation.
   * 
   * @param index
   *          The index to be checked.
   * @throws IndexOutOfBoundsException
   *           If the specified index is not a valid position index.
   */
  private void checkPositionIndex(int index) {
    if (!isPositionIndex(index)) {
      throw new IndexOutOfBoundsException(outOfBoundsMsg(index));
    }
  }

  /**
   * Constructs an IndexOutOfBoundsException detail message.
   * 
   * @param index
   *          The index that is out of bounds.
   * @return An IndexOutOfBoundsException detail message.
   */
  private String outOfBoundsMsg(int index) {
    return "Index: " + index + ", Size: " + size;
  }

  @Override
  public int size() {
    return size;
  }

  @Override
  public boolean isEmpty() {
    return size == 0;
  }

  @Override
  public Iterator<E> iterator() {
    return listIterator();
  }

  @Override
  public ListIterator<E> listIterator() {
    return listIterator(0);
  }

  @Override
  public ListIterator<E> listIterator(int index) {
    checkPositionIndex(index);
    return new LinkedListIterator(index);
  }

  @Override
  public Object[] toArray() {
    Object[] array = new Object[size];
    Node<E> curr = head;
    for (int i = 0; i < size; i++) {
      array[i] = curr.element;
      curr = curr.next;
    }
    return array;
  }

  @SuppressWarnings("unchecked")
  @Override
  public <T> T[] toArray(T[] a) {
    T[] array = a.length >= size ? a : (T[]) java.lang.reflect.Array
        .newInstance(a.getClass().getComponentType(), size);
    Node<E> curr = head;
    for (int i = 0; i < size; i++) {
      array[i] = (T) curr.element;
      curr = curr.next;
    }
    return array;
  }

  @Override
  public String toString() {
    if (size == 0) {
      return "[]";
    }
    StringBuilder sb = new StringBuilder();
    sb.append('[');
    for (Node<E> curr = head;; curr = curr.next) {
      E element = curr.element;
      sb.append(element == this ? "(this Collection)" : element);
      if (curr.next == null) {
        return sb.append(']').toString();
      }
      sb.append(", ");
    }
  }

  @Override
  public int hashCode() {
    int hashCode = 1;
    for (Node<E> curr = head; curr != null; curr = curr.next) {
      hashCode = 31 * hashCode
          + (curr.element == null ? 0 : curr.element.hashCode());
    }
    return hashCode;
  }

  @Override
  public boolean equals(Object o) {
    if (o == this) {
      return true;
    }
    if (!(o instanceof List)) {
      return false;
    }
    Node<E> curr;
    Iterator<?> it = ((List<?>) o).iterator();
    for (curr = head; curr != null && it.hasNext(); curr = curr.next) {
      E element1 = curr.element;
      Object element2 = it.next();
      if (!(element1 == null ? element2 == null : element1.equals(element2))) {
        return false;
      }
    }
    return curr == null && !it.hasNext();
  }

  /**
   * Represents a node in this list.
   */
  private static class Node<E> {

    E element;
    Node<E> prev, next;

    Node(E element, Node<E> prev, Node<E> next) {
      this.element = element;
      this.prev = prev;
      this.next = next;
    }
  }

  /**
   * An iterator over the elements in this list.
   */
  private class LinkedListIterator implements ListIterator<E> {

    private Node<E> lastReturned;
    private Node<E> next;
    private int nextIndex;

    LinkedListIterator(int index) {
      next = (index == size) ? null : getNode(index);
      nextIndex = index;
      lastReturned = null;
    }

    @Override
    public int nextIndex() {
      return nextIndex;
    }

    @Override
    public boolean hasNext() {
      return nextIndex < size;
    }

    @Override
    public E next() {
      if (!hasNext()) {
        throw new NoSuchElementException();
      }
      lastReturned = next;
      next = next.next;
      nextIndex++;
      return lastReturned.element;
    }

    @Override
    public int previousIndex() {
      return nextIndex - 1;
    }

    @Override
    public boolean hasPrevious() {
      return nextIndex > 0;
    }

    @Override
    public E previous() {
      if (!hasPrevious()) {
        throw new NoSuchElementException();
      }
      lastReturned = next = (next == null) ? tail : next.prev;
      nextIndex--;
      return lastReturned.element;
    }

    @Override
    public void add(E element) {
      lastReturned = null;
      if (next == null) {
        linkAtTail(element);
      } else {
        linkBefore(element, next);
      }
      nextIndex++;
    }

    @Override
    public void set(E element) {
      if (lastReturned == null) {
        throw new IllegalStateException();
      }
      lastReturned.element = element;
    }

    @Override
    public void remove() {
      if (lastReturned == null) {
        throw new IllegalStateException();
      }
      Node<E> lastNext = lastReturned.next;
      unlink(lastReturned);
      if (next == lastReturned) {
        next = lastNext;
      } else {
        nextIndex--;
      }
      lastReturned = null;
    }
  }
}
