/**
 * A memory manager for storing variable length data elements in a memory space.
 * Communicates with a memory pool and a free list to determine how best to
 * dynamically allocate memory.
 * 
 * @author Michael D. Naper, Jr. <MichaelNaper.com>
 * @version 2013.01.17
 */
public class MemoryManager {

  // Size of message tag (measured in bytes)
  private static final int TAG_SIZE = 2;

  /**
   * Maximum number of bytes allowable for an allocated memory block.
   */
  public static final int MAXIMUM_BLOCK_SIZE = ((1 << (TAG_SIZE * Byte.SIZE)) - 1)
      - TAG_SIZE;

  // Memory pool for storing data
  private final MemoryPool memoryPool;

  // Free list to keep track of free blocks in the memory pool
  private final FreeList freeList;

  // Length (measured in bytes) by which memory pool will expand, if supported,
  // upon unsuccessful memory allocation
  private final int expansionLength;

  /**
   * Constructs a new {@code MemoryManager} with the specified
   * {@link MemoryPool}, {@link FreeList}, and expansion length (measured in
   * bytes).
   * 
   * @param memoryPool
   *          The memory pool for allowing dynamic memory allocation.
   * @param freeList
   *          The list to keep track of the free blocks in the memory pool.
   * @param expansionLength
   *          The length (measured in bytes) by which the memory pool will
   *          expand, if supported, upon unsuccessful memory allocation. A value
   *          of {@code 0} will result in no attempt at expansion.
   */
  public MemoryManager(MemoryPool memoryPool, FreeList freeList,
      int expansionLength) {
    this.memoryPool = memoryPool;
    this.freeList = freeList;
    this.expansionLength = expansionLength;

    if (this.memoryPool == null) {
      throw new NullPointerException("memoryPool is null.");
    }
    if (this.freeList == null) {
      throw new NullPointerException("freeList is null.");
    }
    if (this.expansionLength < 0) {
      throw new IllegalArgumentException(
          "expansionLength must be a non-negative integer.");
    }

    if (memoryPool.size() > 0) {
      // memory pool is initially all free
      freeList.addBlock(new MemoryHandle(0), memoryPool.size());
    }
  }

  /**
   * Constructs a new {@code MemoryManager} with the specified
   * {@link MemoryPool} and {@link FreeList}. No expansion of the memory pool
   * will be attempted upon unsuccessful memory allocation.
   * 
   * @param memoryPool
   *          The memory pool for allowing dynamic memory allocation.
   * @param freeList
   *          The list to keep track of the free blocks in the memory pool.
   */
  public MemoryManager(MemoryPool memoryPool, FreeList freeList) {
    this(memoryPool, freeList, 0);
  }

  /**
   * Allocates memory for the specified data.
   * 
   * @param data
   *          The data for which to allocate memory.
   * @return A {@link MemoryHandle} to the memory allocated, or {@code null} if
   *         the required memory is unable to be allocated.
   */
  public MemoryHandle allocate(byte[] data) {
    if (data == null) {
      throw new NullPointerException("data is null.");
    }
    if (data.length > MAXIMUM_BLOCK_SIZE) {
      throw new IllegalArgumentException(
          "data length must not exceed maximum block size "
              + MAXIMUM_BLOCK_SIZE + ".");
    }

    int messageSize = TAG_SIZE + data.length;
    MemoryHandle address = freeList.requestBlock(messageSize);
    while (address == null) {
      if (expansionLength > 0 && memoryPool.canExpand(expansionLength)) {
        int oldPoolSize = memoryPool.size();
        memoryPool.expand(expansionLength);
        freeList.addBlock(new MemoryHandle(oldPoolSize), expansionLength);
        address = freeList.requestBlock(messageSize);
      } else {
        return null; // unable to find block of sufficient size
      }
    }

    byte[] tag = Utilities.intToByteArray(messageSize);
    byte[] message = new byte[messageSize];
    System.arraycopy(tag, tag.length - TAG_SIZE, message, 0, TAG_SIZE);
    System.arraycopy(data, 0, message, TAG_SIZE, data.length);
    memoryPool.insert(message, address.getResource());
    return address;
  }

  /**
   * Dereferences the memory specified by the specified {@link MemoryHandle}.
   * 
   * @param handle
   *          A memory handle to the memory that is to be dereferenced.
   * @return The data corresponding to the dereferenced memory.
   */
  public byte[] dereference(MemoryHandle handle) {
    if (handle == null) {
      throw new NullPointerException("handle is null.");
    }

    int dataAddress = handle.getResource() + TAG_SIZE;
    int dataSize = extractMessageSize(handle) - TAG_SIZE;
    return memoryPool.retrieve(dataAddress, dataSize);
  }

  /**
   * Frees the allocated memory specified by the specified {@link MemoryHandle}.
   * 
   * @param handle
   *          A memory handle to the allocated memory that is to be freed.
   */
  public void free(MemoryHandle handle) {
    if (handle == null) {
      throw new NullPointerException("handle is null.");
    }

    freeList.addBlock(handle, extractMessageSize(handle));
  }

  /**
   * Returns the message size extracted from the memory specified by the
   * specified {@link MemoryHandle}.
   * 
   * @param handle
   *          A {@link MemoryHandle} to the memory from which the message size
   *          is to be extracted.
   * @return The message size extracted from the memory specified by the
   *         specified {@link MemoryHandle}.
   */
  private int extractMessageSize(MemoryHandle handle) {
    assert handle != null : "handle is null.";

    byte[] tag = memoryPool.retrieve(handle.getResource(), TAG_SIZE);
    byte[] messageSize = new byte[Utilities.INT_BYTE_SIZE];
    System.arraycopy(tag, 0, messageSize, messageSize.length - TAG_SIZE,
        TAG_SIZE);
    return Utilities.byteArrayToInt(messageSize);
  }

  /**
   * Returns a string representation of the free list.
   * 
   * @return A string representation of the free list.
   */
  public String freeListToString() {
    return freeList.toString();
  }
}
