import java.io.IOException;

/**
 * A memory pool for storing data.
 * 
 * @author Michael D. Naper, Jr. <MichaelNaper.com>
 * @version 2013.01.17
 */
public class MemoryPool {

  // Buffer pool buffering the file in which the data are stored
  private final BufferPool pool;

  // Size (capacity) of the memory pool (measured in bytes)
  private int poolSize;

  /**
   * Constructs a new {@code MemoryPool} backed by the specified
   * {@link BufferPool} and with the specified initial pool size, measured in
   * bytes.
   * 
   * @param bufferPool
   *          The buffer pool buffering the file in which the data are stored.
   * @param poolSize
   *          The initial size of the memory pool, measured in bytes.
   */
  public MemoryPool(BufferPool bufferPool, int poolSize) {
    pool = bufferPool;
    this.poolSize = poolSize;

    if (bufferPool == null) {
      throw new NullPointerException("bufferPool is null.");
    }
    if (this.poolSize < 0) {
      throw new IllegalArgumentException(
          "poolSize must be a non-negative integer.");
    }
  }

  /**
   * Inserts the specified data into the memory pool starting at the specified
   * address.
   * 
   * @param data
   *          The data to insert into the memory pool.
   * @param address
   *          The address in the memory pool at which to insert the data.
   */
  public void insert(byte[] data, int address) {
    if (data == null) {
      throw new NullPointerException("data is null.");
    }
    checkMemoryBounds(address, data.length);

    try {
      pool.write(data, address);
    } catch (IOException e) {
      System.err.println("An I/O error has occurred.");
    }
  }

  /**
   * Retrieves the data starting at the specified address up to the specified
   * number of bytes.
   * 
   * @param address
   *          The address in the memory pool at which to retrieve the data.
   * @param length
   *          The number of bytes to retrieve.
   * @return The retrieved data.
   */
  public byte[] retrieve(int address, int length) {
    checkMemoryBounds(address, length);

    try {
      return pool.read(address, length);
    } catch (IOException e) {
      System.err.println("An I/O error has occurred.");
      return null;
    }
  }

  /**
   * Checks whether the specified address and length refer to memory within the
   * bounds of this memory pool.
   * 
   * @param address
   *          The starting address of the memory segment.
   * @param length
   *          The length (measured in bytes) of the memory segment.
   * @throws IllegalArgumentException
   *           If the specified address and length refer to memory out of
   *           bounds.
   */
  private void checkMemoryBounds(int address, int length) {
    if (length <= 0) {
      throw new IllegalArgumentException("length must be a positive integer.");
    }
    if (address < 0 || address + length > poolSize) {
      throw new IllegalArgumentException("Memory out of bounds. [address: "
          + address + ", length: " + length + ", pool size: " + poolSize + "]");
    }
  }

  /**
   * Returns the size of this memory pool, measured in bytes.
   * 
   * @return The size of this memory pool, measured in bytes.
   */
  public int size() {
    return poolSize;
  }

  /**
   * Returns {@code true} if this memory pool can be expanded by the specified
   * length, measured in bytes.
   * 
   * @param length
   *          The length (measured in bytes) to check for the capability to
   *          expand.
   * @return {@code true} if this memory pool can be expanded by the specified
   *         length.
   */
  public boolean canExpand(int length) {
    if (length <= 0) {
      throw new IllegalArgumentException("length must be a positive integer.");
    }

    return true;
  }

  /**
   * Expands this memory pool by the specified length, measured in bytes.
   * 
   * @param length
   *          The length (measured in bytes) by which this memory pool is to be
   *          expanded.
   * @return The new size of this memory pool, measured in bytes.
   * @throws IllegalStateException
   *           If this memory pool cannot be expanded by the specified length.
   */
  public int expand(int length) {
    if (!canExpand(length)) {
      throw new IllegalStateException(
          "Memory pool cannot be expanded by specified length.");
    }

    poolSize += length;
    return poolSize;
  }
}
