import java.util.Collection;

/**
 * A record database for storing and managing records.
 * 
 * @author Michael D. Naper, Jr. <MichaelNaper.com>
 * @version 2013.01.17
 */
public class RecordDatabase {

  // Set for storing DNA strings
  private final DnaTreeSet dnaTreeSet;

  /**
   * Constructs a new {@code RecordDatabase} backed by the specified
   * {@link DnaTreeSet}.
   * 
   * @param dnaTreeSet
   *          The set for storing DNA strings.
   */
  public RecordDatabase(DnaTreeSet dnaTreeSet) {
    this.dnaTreeSet = dnaTreeSet;

    if (this.dnaTreeSet == null) {
      throw new NullPointerException("dnaTreeSet is null.");
    }
  }

  /**
   * Inserts the specified {@link DnaString} record into this database if it is
   * not already present. If this database already contains the record, the call
   * leaves the database unchanged.
   * 
   * @param record
   *          The record to be inserted in this database.
   * @return The depth in the underlying {@link DnaTreeSet} at which the
   *         specified record was added, or {@code -1} if the set already
   *         contained the specified record.
   */
  public int insert(DnaString record) {
    if (record == null) {
      throw new NullPointerException("record is null.");
    }

    return dnaTreeSet.add(record);
  }

  /**
   * Adds all {@link DnaString} records contained in this database matching the
   * specified {@link DnaStringDescriptor} to the specified collection.
   * 
   * @param dnaStringDescriptor
   *          The DNA string descriptor against which to match.
   * @param collection
   *          The collection to which to add records.
   * @return The number of tree nodes in the underlying {@link DnaTreeSet}
   *         visited during the search.
   */
  public int search(DnaStringDescriptor dnaStringDescriptor,
      Collection<DnaString> collection) {
    if (dnaStringDescriptor == null) {
      throw new NullPointerException("dnaStringDescriptor is null.");
    }
    if (collection == null) {
      throw new NullPointerException("collection is null.");
    }

    return dnaTreeSet.search(dnaStringDescriptor, collection);
  }

  /**
   * Removes the specified {@link DnaString} record from this database if it is
   * present.
   * 
   * @param record
   *          The record to be removed from this database, if present.
   * @return {@code true} if this database contained the specified record.
   */
  public boolean remove(DnaString record) {
    if (record == null) {
      throw new NullPointerException("record is null.");
    }

    return dnaTreeSet.remove(record);
  }

  /**
   * Returns a string representation of this database's underlying
   * {@link DnaTreeSet}. This representation includes both the tree's node
   * structure and the records it contains, traversed in a pre-order fashion.
   * 
   * @return A string representation of the underlying {@code DnaTreeSet}.
   */
  public String dnaTreeToString() {
    return dnaTreeSet.toString();
  }

  /**
   * Returns a string representation of this database's underlying
   * {@link DnaTreeSet} with each {@link DnaString} record's length included.
   * This representation includes both the tree's node structure and the records
   * it contains, traversed in a pre-order fashion.
   * 
   * @return A string representation of the underlying {@code DnaTreeSet} with
   *         each record's length included.
   */
  public String dnaTreeToStringWithLengths() {
    return dnaTreeSet.toStringWithLengths();
  }

  /**
   * Returns a string representation of this database's underlying
   * {@link DnaTreeSet} with each {@link DnaString} record's statistics
   * included. This representation includes both the tree's node structure and
   * the records it contains, traversed in a pre-order fashion.
   * 
   * @return A string representation of the underlying {@code DnaTreeSet} with
   *         each record's statistics included.
   */
  public String dnaTreeToStringWithStatistics() {
    return dnaTreeSet.toStringWithStatistics();
  }

  @Override
  public String toString() {
    return "DNA Tree:\n" + dnaTreeSet;
  }
}
