/**
 * Defines a set of methods that perform common, often re-used procedures.
 * 
 * @author Michael D. Naper, Jr. <MichaelNaper.com>
 * @version 2013.01.17
 */
public final class Utilities {

  /**
   * Number of bytes used to represent an int.
   */
  public static final int INT_BYTE_SIZE = (int) Math.ceil((double) Integer.SIZE
      / Byte.SIZE);

  private Utilities() {
    // disable instantiation
    throw new UnsupportedOperationException();
  }

  /**
   * Converts an integer value to its byte array representation.
   * 
   * @param value
   *          The integer to convert.
   * @return The byte array representation.
   */
  public static final byte[] intToByteArray(int value) {
    byte[] bytes = new byte[INT_BYTE_SIZE];
    int byteMask = (1 << Byte.SIZE) - 1;
    for (int i = INT_BYTE_SIZE - 1; i >= 0; i--) {
      bytes[i] = (byte) (value & byteMask);
      value >>= Byte.SIZE;
    }
    return bytes;
  }

  /**
   * Converts a byte array to its integer value representation.
   * 
   * @param bytes
   *          The byte array to convert.
   * @return The integer representation.
   */
  public static final int byteArrayToInt(final byte[] bytes) {
    if (bytes == null) {
      throw new NullPointerException("bytes is null.");
    }
    if (bytes.length != INT_BYTE_SIZE) {
      throw new IllegalArgumentException("bytes is not " + INT_BYTE_SIZE
          + " bytes.");
    }

    int value = 0;
    int byteMask = (1 << Byte.SIZE) - 1;
    for (int i = 0; i < INT_BYTE_SIZE; i++) {
      value <<= Byte.SIZE;
      value |= bytes[i] & byteMask;
    }
    return value;
  }
}
