import static org.junit.Assert.*;

import java.util.List;
import java.util.NoSuchElementException;

import org.junit.Before;
import org.junit.Test;

/**
 * @author Michael D. Naper, Jr. <MichaelNaper.com>
 * @version 2013.01.17
 */
public class CircularListIteratorTest {

  private CircularListIterator<Integer> iter;

  @Before
  public void setUp() throws Exception {}

  @Test
  public void testCircularListIteratorList() {
    List<Integer> list = new LinkedList<>();
    iter = new CircularListIterator<>(list);
    assertNotNull("iterator should not return null.", iter);
    assertEquals("iterator should return 0 as next index.", 0, iter.nextIndex());
  }

  @Test
  public void testCircularListIteratorListInt() {
    List<Integer> list = new LinkedList<>();
    list.add(0);
    list.add(1);
    iter = new CircularListIterator<>(list, 1);
    assertNotNull("iterator should not return null.", iter);
    assertEquals("iterator should return 1 as next index.", 1, iter.nextIndex());
  }

  @Test
  public void testNextIndex() {
    List<Integer> list = new LinkedList<>();
    iter = new CircularListIterator<>(list);
    assertEquals("nextIndex should return 0.", 0, iter.nextIndex());
    list.add(0);
    iter = new CircularListIterator<>(list);
    assertEquals("nextIndex should return 0.", 0, iter.nextIndex());
    iter.next();
    assertEquals("nextIndex should return 0.", 0, iter.nextIndex());
    iter.next();
    assertEquals("nextIndex should return 0.", 0, iter.nextIndex());
    list.add(1);
    iter = new CircularListIterator<>(list);
    assertEquals("nextIndex should return 0.", 0, iter.nextIndex());
    iter.next();
    assertEquals("nextIndex should return 1.", 1, iter.nextIndex());
    iter.next();
    assertEquals("nextIndex should return 0.", 0, iter.nextIndex());
  }

  @Test
  public void testHasNext() {
    List<Integer> list = new LinkedList<>();
    iter = new CircularListIterator<>(list);
    assertFalse("hasNext should return false.", iter.hasNext());
    list.add(0);
    iter = new CircularListIterator<>(list);
    assertTrue("hasNext should return true.", iter.hasNext());
    iter.next();
    assertTrue("hasNext should return true.", iter.hasNext());
    iter.next();
    assertTrue("hasNext should return true.", iter.hasNext());
    list.add(1);
    iter = new CircularListIterator<>(list);
    assertTrue("hasNext should return true.", iter.hasNext());
    iter.next();
    assertTrue("hasNext should return true.", iter.hasNext());
    iter.next();
    assertTrue("hasNext should return true.", iter.hasNext());
  }

  @Test
  public void testNext() {
    List<Integer> list = new LinkedList<>();
    iter = new CircularListIterator<>(list);
    try {
      iter.next();
      fail("next should have thrown NoSuchElementException.");
    } catch (NoSuchElementException e) {
      // test passed
    }
    list.add(0);
    iter = new CircularListIterator<>(list);
    assertEquals("next should return element 0.", 0, (int) iter.next());
    assertEquals("next should return element 0.", 0, (int) iter.next());
    assertEquals("next should return element 0.", 0, (int) iter.next());
    list.add(1);
    iter = new CircularListIterator<>(list);
    assertEquals("next should return element 0.", 0, (int) iter.next());
    assertEquals("next should return element 1.", 1, (int) iter.next());
    assertEquals("next should return element 0.", 0, (int) iter.next());
  }

  @Test
  public void testPreviousIndex() {
    List<Integer> list = new LinkedList<>();
    iter = new CircularListIterator<>(list);
    assertEquals("previousIndex should return 0.", 0, iter.previousIndex());
    list.add(0);
    iter = new CircularListIterator<>(list);
    assertEquals("previousIndex should return 0.", 0, iter.previousIndex());
    iter.previous();
    assertEquals("previousIndex should return 0.", 0, iter.previousIndex());
    iter.previous();
    assertEquals("previousIndex should return 0.", 0, iter.previousIndex());
    list.add(1);
    iter = new CircularListIterator<>(list);
    assertEquals("previousIndex should return 1.", 1, iter.previousIndex());
    iter.previous();
    assertEquals("previousIndex should return 0.", 0, iter.previousIndex());
    iter.previous();
    assertEquals("previousIndex should return 1.", 1, iter.previousIndex());
  }

  @Test
  public void testHasPrevious() {
    List<Integer> list = new LinkedList<>();
    iter = new CircularListIterator<>(list);
    assertFalse("hasPrevious should return false.", iter.hasPrevious());
    list.add(0);
    iter = new CircularListIterator<>(list);
    assertTrue("hasPrevious should return true.", iter.hasPrevious());
    iter.previous();
    assertTrue("hasPrevious should return true.", iter.hasPrevious());
    iter.previous();
    assertTrue("hasPrevious should return true.", iter.hasPrevious());
    list.add(1);
    iter = new CircularListIterator<>(list);
    assertTrue("hasPrevious should return true.", iter.hasPrevious());
    iter.previous();
    assertTrue("hasPrevious should return true.", iter.hasPrevious());
    iter.previous();
    assertTrue("hasPrevious should return true.", iter.hasPrevious());
  }

  @Test
  public void testPrevious() {
    List<Integer> list = new LinkedList<>();
    iter = new CircularListIterator<>(list);
    try {
      iter.previous();
      fail("previous should have thrown NoSuchElementException.");
    } catch (NoSuchElementException e) {
      // test passed
    }
    list.add(0);
    iter = new CircularListIterator<>(list);
    assertEquals("previous should return element 0.", 0, (int) iter.previous());
    assertEquals("previous should return element 0.", 0, (int) iter.previous());
    assertEquals("previous should return element 0.", 0, (int) iter.previous());
    list.add(1);
    iter = new CircularListIterator<>(list);
    assertEquals("previous should return element 1.", 1, (int) iter.previous());
    assertEquals("previous should return element 0.", 0, (int) iter.previous());
    assertEquals("previous should return element 1.", 1, (int) iter.previous());
  }

  @Test
  public void testAdd() {
    List<Integer> list = new LinkedList<>();
    iter = new CircularListIterator<>(list);
    iter.add(0);
    // [0:0|]
    assertEquals("size should be 1.", 1, list.size());
    assertEquals("nextIndex should return 0.", 0, iter.nextIndex());
    assertTrue("hasNext should return true.", iter.hasNext());
    iter.previous();
    iter.add(1);
    // [1, 0:1,|0]
    assertEquals("size should be 2.", 2, list.size());
    assertEquals("nextIndex should return 1.", 1, iter.nextIndex());
    assertTrue("hasNext should return true.", iter.hasNext());
    iter.previous();
    iter.previous();
    iter.add(2);
    // [1, 2,|0:1, 2, 0]
    assertEquals("size should be 3.", 3, list.size());
    assertEquals("nextIndex should return 2.", 2, iter.nextIndex());
    assertTrue("hasNext should return true.", iter.hasNext());
    iter.next();
    iter.add(3);
    // [1, 2, 0, 3|:1, 2, 0, 3]
    assertEquals("size should be 4.", 4, list.size());
    assertEquals("nextIndex should return 0.", 0, iter.nextIndex());
    assertTrue("hasNext should return true.", iter.hasNext());

    assertEquals("element should be element 1", 1, (int) iter.next());
    assertEquals("nextIndex should return 1.", 1, iter.nextIndex());
    assertEquals("element should be element 2", 2, (int) iter.next());
    assertEquals("nextIndex should return 2.", 2, iter.nextIndex());
    assertEquals("element should be element 0", 0, (int) iter.next());
    assertEquals("nextIndex should return 3.", 3, iter.nextIndex());
    assertEquals("element should be element 3", 3, (int) iter.next());
    assertEquals("nextIndex should return 0.", 0, iter.nextIndex());
    assertEquals("element should be element 1", 1, (int) iter.next());
    assertEquals("nextIndex should return 1.", 1, iter.nextIndex());
  }

  @Test
  public void testSet() {
    List<Integer> list = new LinkedList<>();
    iter = new CircularListIterator<>(list);
    try {
      iter.set(0);
      fail("set should have thrown IllegalStateException.");
    } catch (IllegalStateException e) {
      // test passed
    }
    list.add(0);
    iter = new CircularListIterator<>(list);
    try {
      iter.set(1);
      fail("set should have thrown IllegalStateException.");
    } catch (IllegalStateException e) {
      // test passed
    }
    iter.next();
    iter.set(1);
    assertEquals("size should be 1.", 1, list.size());
    assertEquals("nextIndex should return 0.", 0, iter.nextIndex());
    assertEquals("element should have been set to 1.", 1, (int) iter.previous());
    assertEquals("nextIndex should return 0.", 0, iter.nextIndex());
    assertEquals("element should have been set to 1.", 1, (int) iter.previous());
    iter.set(2);
    assertEquals("size should be 1.", 1, list.size());
    assertEquals("nextIndex should return 0.", 0, iter.nextIndex());
    assertEquals("element should have been set to 2.", 2, (int) iter.next());
    assertEquals("nextIndex should return 0.", 0, iter.nextIndex());
    assertEquals("element should have been set to 2.", 2, (int) iter.next());
  }

  @Test
  public void testRemove() {
    List<Integer> list = new LinkedList<>();
    list.add(0);
    iter = new CircularListIterator<>(list);
    iter.next();
    iter.remove();
    assertEquals("size should be 0.", 0, list.size());
    assertEquals("nextIndex should return 0.", 0, iter.nextIndex());
    assertFalse("hasNext should return false.", iter.hasNext());
    list.add(0);
    iter = new CircularListIterator<>(list);
    iter.previous();
    iter.remove();
    assertEquals("size should be 0.", 0, list.size());
    assertEquals("nextIndex should return 0.", 0, iter.nextIndex());
    assertFalse("hasNext should return false.", iter.hasNext());
    list.add(0);
    list.add(1);
    iter = new CircularListIterator<>(list);
    iter.next();
    iter.remove();
    assertEquals("size should be 1.", 1, list.size());
    assertEquals("nextIndex should return 0.", 0, iter.nextIndex());
    assertTrue("hasNext should return true.", iter.hasNext());
    assertEquals("element should be 1.", 1, (int) iter.next());
    iter.previous();
    assertEquals("element should be 1.", 1, (int) iter.previous());
  }

}
