import static org.junit.Assert.*;

import java.math.BigInteger;
import java.util.Arrays;

import org.junit.Before;
import org.junit.Test;

/**
 * @author Michael D. Naper, Jr. <MichaelNaper.com>
 * @version 2013.01.17
 */
public class DnaStringSerializerTest {

  private Serializer<DnaString> serializer;

  @Before
  public void setUp() throws Exception {
    serializer = DnaStringSerializer.getInstance();
  }

  @Test
  public void testGetInstance() {
    assertNotNull("serializer should not be null.", serializer);
  }

  @Test
  public void testSerialize() {
    assertEquals("A should serialize to {00000000, 00000001, 00000000}.",
        "100000000", binaryString(serializer.serialize(new DnaString("A"))));
    assertEquals("C should serialize to {00000000, 00000001, 0100000}.",
        "101000000", binaryString(serializer.serialize(new DnaString("C"))));
    assertEquals("G should serialize to {00000000, 00000001, 10000000}.",
        "110000000", binaryString(serializer.serialize(new DnaString("G"))));
    assertEquals("T should serialize to {00000000, 00000001, 11000000}.",
        "111000000", binaryString(serializer.serialize(new DnaString("T"))));

    assertEquals("AA should serialize to {00000000, 00000010, 00000000}.",
        "1000000000", binaryString(serializer.serialize(new DnaString("AA"))));
    assertEquals("CC should serialize to {00000000, 00000010, 01010000}.",
        "1001010000", binaryString(serializer.serialize(new DnaString("CC"))));
    assertEquals("GG should serialize to {00000000, 00000010, 10100000}.",
        "1010100000", binaryString(serializer.serialize(new DnaString("GG"))));
    assertEquals("TT should serialize to {00000000, 00000010, 11110000}.",
        "1011110000", binaryString(serializer.serialize(new DnaString("TT"))));

    assertEquals("AAAA should serialize to {00000000, 00000100, 00000000}.",
        "10000000000",
        binaryString(serializer.serialize(new DnaString("AAAA"))));
    assertEquals("CCCC should serialize to {00000000, 00000100, 01010101}.",
        "10001010101",
        binaryString(serializer.serialize(new DnaString("CCCC"))));
    assertEquals("GGGG should serialize to {00000000, 00000100, 10101010}.",
        "10010101010",
        binaryString(serializer.serialize(new DnaString("GGGG"))));
    assertEquals("TTTT should serialize to {00000000, 00000100, 11111111}.",
        "10011111111",
        binaryString(serializer.serialize(new DnaString("TTTT"))));

    assertEquals("ACGT should serialize to {00000000, 00000100, 00011011}.",
        "10000011011",
        binaryString(serializer.serialize(new DnaString("ACGT"))));
    assertEquals("TGCA should serialize to {00000000, 00000100, 11100100}.",
        "10011100100",
        binaryString(serializer.serialize(new DnaString("TGCA"))));

    assertEquals(
        "AAAAA should serialize to {00000000, 00000101, 00000000, 00000000}.",
        "1010000000000000000",
        binaryString(serializer.serialize(new DnaString("AAAAA"))));
    assertEquals(
        "CCCCC should serialize to {00000000, 00000101, 01010101, 01000000}.",
        "1010101010101000000",
        binaryString(serializer.serialize(new DnaString("CCCCC"))));
    assertEquals(
        "GGGGG should serialize to {00000000, 00000101, 10101010, 10000000}.",
        "1011010101010000000",
        binaryString(serializer.serialize(new DnaString("GGGGG"))));
    assertEquals(
        "TTTTT should serialize to {00000000, 00000101, 11111111, 11000000}.",
        "1011111111111000000",
        binaryString(serializer.serialize(new DnaString("TTTTT"))));

    char[] longStringChars = new char[DnaStringSerializer.MAX_LENGTH];
    Arrays.fill(longStringChars, 'A');
    String longString = new String(longStringChars);
    try {
      serializer.serialize(new DnaString(longString));
      // test passed
    } catch (IllegalArgumentException e) {
      fail("serailize should not have thrown IllegalArgumentException.");
    }

    longStringChars = new char[DnaStringSerializer.MAX_LENGTH + 1];
    Arrays.fill(longStringChars, 'A');
    longString = new String(longStringChars);
    try {
      serializer.serialize(new DnaString(longString));
      fail("serialize should have thrown IllegalArgumentException.");
    } catch (IllegalArgumentException e) {
      // test passed
    }
  }

  @Test
  public void testDeserialize() {
    try {
      serializer.deserialize(new byte[] { 0b00000000 });
      fail("deserialize should have thrown InvalidBytesException.");
    } catch (Serializer.InvalidBytesException e) {
      // test passed
    }
    try {
      assertEquals(
          "{00000000, 00000001, 00000000} should have deserialized to A.",
          new DnaString("A"),
          serializer.deserialize(new byte[] { 0b00000000, 0b00000001,
              0b00000000 }));
      assertEquals(
          "{00000000, 00000001, 01000000} should have deserialized to C.",
          new DnaString("C"),
          serializer.deserialize(new byte[] { 0b00000000, 0b00000001,
              0b01000000 }));
      assertEquals(
          "{00000000, 00000001, 10000000} should have deserialized to G.",
          new DnaString("G"),
          serializer.deserialize(new byte[] { 0b00000000, 0b00000001,
              (byte) 0b10000000 }));
      assertEquals(
          "{00000000, 00000001, 11000000} should have deserialized to T.",
          new DnaString("T"),
          serializer.deserialize(new byte[] { 0b00000000, 0b00000001,
              (byte) 0b11000000 }));

      assertEquals(
          "{00000000, 00000010, 00000000} should have deserialized to AA.",
          new DnaString("AA"),
          serializer.deserialize(new byte[] { 0b00000000, 0b00000010,
              0b00000000 }));
      assertEquals(
          "{00000000, 00000010, 01010000} should have deserialized to CC.",
          new DnaString("CC"),
          serializer.deserialize(new byte[] { 0b00000000, 0b00000010,
              0b01010000 }));
      assertEquals(
          "{00000000, 00000010, 10100000} should have deserialized to GG.",
          new DnaString("GG"),
          serializer.deserialize(new byte[] { 0b00000000, 0b00000010,
              (byte) 0b10100000 }));
      assertEquals(
          "{00000000, 00000010, 11110000} should have deserialized to TT.",
          new DnaString("TT"),
          serializer.deserialize(new byte[] { 0b00000000, 0b00000010,
              (byte) 0b11110000 }));

      assertEquals(
          "{00000000, 00000100, 00000000} should have deserialized to AAAA.",
          new DnaString("AAAA"),
          serializer.deserialize(new byte[] { 0b00000000, 0b00000100,
              0b00000000 }));
      assertEquals(
          "{00000000, 00000100, 01010101} should have deserialized to CCCC.",
          new DnaString("CCCC"),
          serializer.deserialize(new byte[] { 0b00000000, 0b00000100,
              0b01010101 }));
      assertEquals(
          "{00000000, 00000100, 10101010} should have deserialized to GGGG.",
          new DnaString("GGGG"),
          serializer.deserialize(new byte[] { 0b00000000, 0b00000100,
              (byte) 0b10101010 }));
      assertEquals(
          "{00000000, 00000100, 11111111} should have deserialized to TTTT.",
          new DnaString("TTTT"),
          serializer.deserialize(new byte[] { 0b00000000, 0b00000100,
              (byte) 0b11111111 }));

      assertEquals(
          "{00000000, 00000100, 00011011} should have deserialized to ACGT.",
          new DnaString("ACGT"),
          serializer.deserialize(new byte[] { 0b00000000, 0b00000100,
              0b00011011 }));
      assertEquals(
          "{00000000, 00000100, 11100100} should have deserialized to TGCA.",
          new DnaString("TGCA"),
          serializer.deserialize(new byte[] { 0b00000000, 0b00000100,
              (byte) 0b11100100 }));

      assertEquals(
          "{00000000, 00000101, 00000000, 00000000} should have deserialized to AAAAA.",
          new DnaString("AAAAA"),
          serializer.deserialize(new byte[] { 0b00000000, 0b00000101,
              0b00000000, 0b00000000 }));
      assertEquals(
          "{00000000, 00000101, 01010101, 01000000} should have deserialized to CCCCC.",
          new DnaString("CCCCC"),
          serializer.deserialize(new byte[] { 0b00000000, 0b00000101,
              0b01010101, 0b01000000 }));
      assertEquals(
          "{00000000, 00000101, 10101010, 10000000} should have deserialized to GGGGG.",
          new DnaString("GGGGG"),
          serializer.deserialize(new byte[] { 0b00000000, 0b00000101,
              (byte) 0b10101010, (byte) 0b10000000 }));
      assertEquals(
          "{00000000, 00000101, 11111111, 11000000} should have deserialized to TTTTT.",
          new DnaString("TTTTT"),
          serializer.deserialize(new byte[] { 0b00000000, 0b00000101,
              (byte) 0b11111111, (byte) 0b11000000 }));
    } catch (Serializer.InvalidBytesException e) {
      fail("deserialize should not have thrown InvalidBytesException.");
    }
  }

  private static String binaryString(byte[] val) {
    return new BigInteger(1, val).toString(2);
  }
}
