import static org.junit.Assert.*;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.junit.Before;
import org.junit.Test;

/**
 * @author Michael D. Naper, Jr. <MichaelNaper.com>
 * @version 2013.01.17
 */
public class FreeListTest {

  private static final Pattern FREE_BLOCK_REGEX = Pattern
      .compile("\\[([0-9]+), [0-9]+\\] \\(([0-9]+) bytes\\)");

  private FreeList freeList;

  private static class Block {
    int address, size;

    Block(int address, int size) {
      this.address = address;
      this.size = size;
    }

    @Override
    public String toString() {
      return "[address: " + address + ", size: " + size + "]";
    }

    @Override
    public boolean equals(Object obj) {
      Block other = (Block) obj;
      return address == other.address && size == other.size;
    }
  }

  private Block B(int address, int size) {
    return new Block(address, size);
  }

  private List<Block> freeListBlocks() {
    List<Block> blocks = new ArrayList<>();
    Matcher matcher = FREE_BLOCK_REGEX.matcher(freeList.toString());
    while (matcher.find()) {
      int address = Integer.parseInt(matcher.group(1));
      int size = Integer.parseInt(matcher.group(2));
      blocks.add(new Block(address, size));
    }
    return blocks;
  }

  /**
   * Asserts that the free list holds blocks equal to the specified blocks in
   * the specified order.
   * 
   * @param blocks
   *          The blocks expected in the free list.
   */
  private void assertFreeListEquals(Block... blocks) {
    List<Block> freeListBlocks = freeListBlocks();
    if (freeListBlocks.size() < blocks.length) {
      fail("free list has too few blocks.");
      return;
    }
    if (freeListBlocks.size() > blocks.length) {
      fail("free list has too many blocks.");
      return;
    }
    for (int i = 0; i < blocks.length; i++) {
      Block expectedBlock = blocks[i];
      Block actualBlock = freeListBlocks.get(i);
      assertEquals("free list is not structured as expected.", expectedBlock,
          actualBlock);
    }
  }

  /**
   * @throws java.lang.Exception
   */
  @Before
  public void setUp() throws Exception {
    freeList = new FreeList();
  }

  /**
   * Test method for {@link FreeList#FreeList()}.
   */
  @Test
  public void testFreeList() {
    assertNotNull(freeList);
  }

  /**
   * Test method for {@link FreeList#addBlock(MemoryHandle, int)}.
   */
  @Test
  public void testAddBlock() {
    try {
      freeList.addBlock(new MemoryHandle(0), -1);
      fail("addBlock should have thrown IllegalArgumentException.");
    } catch (IllegalArgumentException e) {
      // test passed
    }
    try {
      freeList.addBlock(new MemoryHandle(0), 0);
      fail("addBlock should have thrown IllegalArgumentException.");
    } catch (IllegalArgumentException e) {
      // test passed
    }
    freeList.addBlock(new MemoryHandle(0), 5);
    assertFreeListEquals(B(0, 5));
    freeList.addBlock(new MemoryHandle(20), 5);
    assertFreeListEquals(B(0, 5), B(20, 5));
    freeList.addBlock(new MemoryHandle(5), 5);
    assertFreeListEquals(B(0, 10), B(20, 5));
    freeList.addBlock(new MemoryHandle(15), 5);
    assertFreeListEquals(B(0, 10), B(15, 10));
    freeList.addBlock(new MemoryHandle(10), 5);
    assertFreeListEquals(B(0, 25));
  }

  /**
   * Test method for {@link FreeList#requestBlock(int)}.
   */
  @Test
  public void testRequestBlock() {
    assertNull("requestBlock should return null.", freeList.requestBlock(1));
    MemoryHandle addressZero = new MemoryHandle(0);
    freeList.addBlock(addressZero, 5);
    assertNull("requestBlock should return null.", freeList.requestBlock(6));
    assertFreeListEquals(B(0, 5));
    assertEquals("requestBlock should return address 0.", addressZero,
        freeList.requestBlock(5));
    assertFreeListEquals();
    freeList.addBlock(addressZero, 5);
    assertEquals("requestBlock should return address 0.", addressZero,
        freeList.requestBlock(1));
    assertFreeListEquals(B(1, 4));
    assertNull("requestBlock should return null.", freeList.requestBlock(5));
    assertFreeListEquals(B(1, 4));
    assertEquals("requestBlock should return address 1.", new MemoryHandle(1),
        freeList.requestBlock(4));
    assertFreeListEquals();
    freeList.addBlock(new MemoryHandle(14), 5);
    freeList.addBlock(new MemoryHandle(9), 4);
    freeList.addBlock(new MemoryHandle(5), 3);
    freeList.addBlock(new MemoryHandle(2), 2);
    freeList.addBlock(addressZero, 1);
    assertEquals("requestBlock should return address 5.", new MemoryHandle(5),
        freeList.requestBlock(3));
    assertFreeListEquals(B(0, 1), B(2, 2), B(9, 4), B(14, 5));
    assertEquals("requestBlock should return address 9.", new MemoryHandle(9),
        freeList.requestBlock(1));
    assertFreeListEquals(B(0, 1), B(2, 2), B(10, 3), B(14, 5));
    assertEquals("requestBlock should return address 14.",
        new MemoryHandle(14), freeList.requestBlock(4));
    assertFreeListEquals(B(0, 1), B(2, 2), B(10, 3), B(18, 1));
    assertEquals("requestBlock should return address 10.",
        new MemoryHandle(10), freeList.requestBlock(3));
    assertFreeListEquals(B(0, 1), B(2, 2), B(18, 1));
    freeList.addBlock(new MemoryHandle(20), 2);
    assertEquals("requestBlock should return address 20.",
        new MemoryHandle(20), freeList.requestBlock(1));
    assertFreeListEquals(B(0, 1), B(2, 2), B(18, 1), B(21, 1));
  }

  /**
   * Test method for {@link FreeList#toString()}.
   */
  @Test
  public void testToString() {
    freeList.addBlock(new MemoryHandle(5), 3);
    freeList.addBlock(new MemoryHandle(9), 4);
    freeList.addBlock(new MemoryHandle(2), 2);
    freeList.addBlock(new MemoryHandle(14), 5);
    freeList.addBlock(new MemoryHandle(0), 1);
    StringBuilder expected = new StringBuilder();
    expected.append("[[0, 0] (1 bytes), ");
    expected.append("[2, 3] (2 bytes), ");
    expected.append("[5, 7] (3 bytes), ");
    expected.append("[9, 12] (4 bytes), ");
    expected.append("[14, 18] (5 bytes)]");
    assertEquals("toString does not return correct string.",
        expected.toString(), freeList.toString());
  }
}
