import static org.junit.Assert.*;

import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;
import java.util.NoSuchElementException;

import org.junit.Before;
import org.junit.Test;

/**
 * @author Michael D. Naper, Jr. <MichaelNaper.com>
 * @version 2013.01.17
 */
public class LinkedListTest {

  private static final List<Integer> SAMPLE_DATA = generateSampleData();

  private List<Integer> list;

  private static List<Integer> generateSampleData() {
    List<Integer> sampleData = new ArrayList<Integer>();
    for (int i = 0; i < 10; i++) {
      sampleData.add(i);
    }
    return sampleData;
  }

  private static int expectedSampleDataHashCode() {
    return SAMPLE_DATA.hashCode();
  }

  /**
   * @throws java.lang.Exception
   */
  @Before
  public void setUp() throws Exception {
    list = new LinkedList<>();
  }

  /**
   * Test method for {@link LinkedList#LinkedList()}.
   */
  @Test
  public void testLinkedList() {
    assertNotNull(list);
    assertEquals("size should be initally 0.", 0, list.size());
    assertTrue("list should be initially empty.", list.isEmpty());
  }

  /**
   * Test method for {@link LinkedList#LinkedList(java.util.Collection)}.
   */
  @Test
  public void testLinkedListCollection() {
    List<Integer> list = new LinkedList<>(SAMPLE_DATA);
    assertNotNull(list);
    assertEquals("size should be initally the size of sample data.",
        SAMPLE_DATA.size(), list.size());
    if (SAMPLE_DATA.isEmpty()) {
      assertTrue("list should be initially empty.", list.isEmpty());
    } else {
      assertFalse("list should not be initially empty.", list.isEmpty());
    }
  }

  /**
   * Test method for {@link LinkedList#add(java.lang.Object)}.
   */
  @Test
  public void testAddE() {
    assertTrue(list.add(0));
    assertEquals("size should be 1.", 1, list.size());
    assertFalse("list should not be empty.", list.isEmpty());
    assertEquals("head of list should be element 0.", 0, (int) list.get(0));
    assertEquals("tail of list should be element 0.", 0, (int) list.get(0));
    assertTrue(list.add(1));
    assertEquals("size should be 2.", 2, list.size());
    assertEquals("head of list should be element 0.", 0, (int) list.get(0));
    assertEquals("tail of list should be element 1.", 1, (int) list.get(1));
    assertTrue(list.add(2));
    assertEquals("size should be 3.", 3, list.size());
    assertEquals("head of list should be element 0.", 0, (int) list.get(0));
    assertEquals("position 1 of list should be element 1.", 1,
        (int) list.get(1));
    assertEquals("tail of list should be element 2.", 2, (int) list.get(2));
  }

  /**
   * Test method for {@link LinkedList#add(int, java.lang.Object)}.
   */
  @Test
  public void testAddIntE() {
    try {
      list.add(-1, 0);
      fail("add should have thrown IndexOutOfBoundsException.");
    } catch (IndexOutOfBoundsException e) {
      assertEquals("size should be 0.", 0, list.size());
      assertTrue("list should be empty.", list.isEmpty());
    }
    list.add(0, 2);
    assertEquals("size should be 1.", 1, list.size());
    assertFalse("list should not be empty.", list.isEmpty());
    assertEquals("head of list should be element 2.", 2, (int) list.get(0));
    assertEquals("tail of list should be element 2.", 2, (int) list.get(0));
    list.add(1, 3);
    assertEquals("size should be 2.", 2, list.size());
    assertEquals("head of list should be element 2.", 2, (int) list.get(0));
    assertEquals("tail of list should be element 3.", 3, (int) list.get(1));
    list.add(0, 0);
    assertEquals("size should be 3.", 3, list.size());
    assertEquals("head of list should be element 0.", 0, (int) list.get(0));
    assertEquals("position 1 of list should be element 2.", 2,
        (int) list.get(1));
    assertEquals("tail of list should be element 3.", 3, (int) list.get(2));
    list.add(1, 1);
    assertEquals("size should be 4.", 4, list.size());
    assertEquals("head of list should be element 0.", 0, (int) list.get(0));
    assertEquals("position 1 of list should be element 1.", 1,
        (int) list.get(1));
    assertEquals("position 2 of list should be element 2.", 2,
        (int) list.get(2));
    assertEquals("tail of list should be element 3.", 3, (int) list.get(3));
  }

  /**
   * Test method for {@link LinkedList#addAll(java.util.Collection)}.
   */
  @Test
  public void testAddAllCollection() {
    list.add(-1);
    list.addAll(SAMPLE_DATA);
    assertEquals("size should be one more than the size of sample data.",
        1 + SAMPLE_DATA.size(), list.size());
    assertEquals("head of list should be element -1.", -1, (int) list.get(0));
    for (int i = 1; i < SAMPLE_DATA.size() + 1; i++) {
      assertEquals("position " + i + " of list should be element "
          + SAMPLE_DATA.get(i - 1) + ".", SAMPLE_DATA.get(i - 1), list.get(i));
    }
  }

  /**
   * Test method for {@link LinkedList#addAll(int, java.util.Collection)}.
   */
  @Test
  public void testAddAllIntCollection() {
    list.add(-1);
    list.add(-2);
    try {
      list.addAll(-1, SAMPLE_DATA);
      fail("addAll should have thrown IndexOutOfBoundsException.");
    } catch (IndexOutOfBoundsException e) {
      assertEquals("size should be 2.", 2, list.size());
    }
    list.addAll(1, SAMPLE_DATA);
    assertEquals("size should be two more than the size of sample data.",
        2 + SAMPLE_DATA.size(), list.size());
    assertEquals("head of list should be element -1.", -1, (int) list.get(0));
    assertEquals("tail of list should be element -2.", -2,
        (int) list.get(list.size() - 1));
    for (int i = 1; i < SAMPLE_DATA.size() + 1; i++) {
      assertEquals("position " + i + " of list should be element "
          + SAMPLE_DATA.get(i - 1) + ".", SAMPLE_DATA.get(i - 1), list.get(i));
    }
  }

  /**
   * Test method for {@link LinkedList#contains(java.lang.Object)}.
   */
  @Test
  public void testContains() {
    assertFalse("list should not contain element 0.", list.contains(0));
    list.add(0);
    assertTrue("list should contain element 0.", list.contains(0));
  }

  /**
   * Test method for {@link LinkedList#containsAll(java.util.Collection)}.
   */
  @Test
  public void testContainsAll() {
    for (int i = 0; i < SAMPLE_DATA.size() - 1; i++) {
      list.add(SAMPLE_DATA.get(i));
    }
    assertFalse("list should not contain all elements in sample data.",
        list.containsAll(SAMPLE_DATA));
    list.add(0, SAMPLE_DATA.get(SAMPLE_DATA.size() - 1));
    assertTrue("list should contain all elements in sample data.",
        list.containsAll(SAMPLE_DATA));
  }

  /**
   * Test method for {@link LinkedList#indexOf(java.lang.Object)}.
   */
  @Test
  public void testIndexOf() {
    assertEquals("index of element 0 should be -1.", -1, list.indexOf(0));
    list.add(0);
    assertEquals("index of element 0 should be 0.", 0, list.indexOf(0));
    list.add(1);
    assertEquals("index of element 1 should be 1.", 1, list.indexOf(1));
    list.add(0);
    assertEquals("index of element 0 should be 0.", 0, list.indexOf(0));
  }

  /**
   * Test method for {@link LinkedList#lastIndexOf(java.lang.Object)}.
   */
  @Test
  public void testLastIndexOf() {
    assertEquals("index of element 0 should be -1.", -1, list.lastIndexOf(0));
    list.add(0);
    assertEquals("index of element 0 should be 0.", 0, list.lastIndexOf(0));
    list.add(1);
    assertEquals("index of element 1 should be 1.", 1, list.lastIndexOf(1));
    list.add(0);
    assertEquals("index of element 0 should be 0.", 2, list.lastIndexOf(0));
  }

  /**
   * Test method for {@link LinkedList#get(int)}.
   */
  @Test
  public void testGet() {
    try {
      list.get(0);
      fail("get should have thrown IndexOutOfBoundsException.");
    } catch (IndexOutOfBoundsException e) {
      // test passed
    }
    for (int i = 0; i < 4; i++) {
      list.add(i);
    }
    for (int i = 0; i < 4; i++) {
      assertEquals("position " + i + " of list should be " + i + ".", i,
          (int) list.get(i));
    }
  }

  /**
   * Test method for {@link LinkedList#set(int, java.lang.Object)}.
   */
  @Test
  public void testSet() {
    try {
      list.set(0, 0);
      fail("set should have thrown IndexOutOfBoundsException.");
    } catch (IndexOutOfBoundsException e) {
      // test passed
    }
    for (int i = 0; i < 4; i++) {
      list.add(i);
    }
    for (int i = 0; i < 4; i++) {
      assertEquals("element returned should be " + i, i,
          (int) list.set(i, i + 4));
      assertEquals("position " + i + " of list should be " + (i + 4) + ".",
          i + 4, (int) list.get(i));
    }
  }

  /**
   * Test method for {@link LinkedList#remove(java.lang.Object)}.
   */
  @Test
  public void testRemoveObject() {
    assertFalse("remove should have returned false.",
        list.remove(Integer.valueOf(0)));
    list.add(0);
    list.add(1);
    list.add(2);
    assertTrue("remove should have returned true.",
        list.remove(Integer.valueOf(1)));
    assertEquals("size should be 2.", 2, list.size());
    assertEquals("position 0 of list should be 0.", 0, (int) list.get(0));
    assertEquals("position 1 of list should be 2.", 2, (int) list.get(1));
    assertTrue("remove should have returned true.",
        list.remove(Integer.valueOf(0)));
    assertEquals("size should be 1.", 1, list.size());
    assertEquals("position 0 of list should be 2.", 2, (int) list.get(0));
    assertTrue("remove should have returned true.",
        list.remove(Integer.valueOf(2)));
    assertEquals("size should be 0.", 0, list.size());
    assertTrue("list should be empty.", list.isEmpty());
  }

  /**
   * Test method for {@link LinkedList#remove(int)}.
   */
  @Test
  public void testRemoveInt() {
    try {
      list.remove(0);
      fail("remove should have thrown IndexOutOfBoundsException.");
    } catch (IndexOutOfBoundsException e) {
      // test passed
    }
    list.add(0);
    list.add(1);
    list.add(2);
    assertEquals("remove should have returned 1.", 1, (int) list.remove(1));
    assertEquals("size should be 2.", 2, list.size());
    assertEquals("position 0 of list should be 0.", 0, (int) list.get(0));
    assertEquals("position 1 of list should be 2.", 2, (int) list.get(1));
    assertEquals("remove should have returned 0.", 0, (int) list.remove(0));
    assertEquals("size should be 1.", 1, list.size());
    assertEquals("position 0 of list should be 2.", 2, (int) list.get(0));
    assertEquals("remove should have returned 2.", 2, (int) list.remove(0));
    assertEquals("size should be 0.", 0, list.size());
    assertTrue("list should be empty.", list.isEmpty());
  }

  /**
   * Test method for {@link LinkedList#removeAll(java.util.Collection)}.
   */
  @Test
  public void testRemoveAll() {
    list.addAll(SAMPLE_DATA);
    list.add(SAMPLE_DATA.size() >> 2, -1);
    list.removeAll(SAMPLE_DATA);
    assertEquals("size should be 1.", 1, list.size());
    assertEquals("position 0 of list should be element -1", -1,
        (int) list.get(0));
  }

  /**
   * Test method for {@link LinkedList#retainAll(java.util.Collection)}.
   */
  @Test
  public void testRetainAll() {
    list.addAll(SAMPLE_DATA);
    list.add(SAMPLE_DATA.size() >> 2, -1);
    list.retainAll(SAMPLE_DATA);
    assertEquals("size should be size of sample data.", SAMPLE_DATA.size(),
        list.size());
    assertFalse("list should not contain element -1.", list.contains(-1));
  }

  /**
   * Test method for {@link LinkedList#clear()}.
   */
  @Test
  public void testClear() {
    list.add(0);
    list.add(1);
    list.clear();
    assertEquals("size should be 0.", 0, list.size());
    assertTrue("list should be empty.", list.isEmpty());
  }

  /**
   * Test method for {@link LinkedList#subList(int, int)}.
   */
  @Test
  public void testSubList() {
    for (int i = 0; i < 4; i++) {
      list.add(i);
    }
    List<Integer> subList = list.subList(1, 3);
    assertNotNull("subList should not be null.", subList);
    for (int i = 0; i < 2; i++) {
      assertEquals(
          "position " + i + " of subList should be element " + list.get(i + 1),
          list.get(i + 1), subList.get(i));
    }
  }

  /**
   * Test method for {@link LinkedList#size()}.
   */
  @Test
  public void testSize() {
    assertEquals("size should be 0.", 0, list.size());
    list.add(0);
    assertEquals("size should be 1.", 1, list.size());
    list.remove(0);
    assertEquals("size should be 0.", 0, list.size());
  }

  /**
   * Test method for {@link LinkedList#isEmpty()}.
   */
  @Test
  public void testIsEmpty() {
    assertTrue("list should be empty.", list.isEmpty());
    list.add(0);
    assertFalse("list should not be empty.", list.isEmpty());
    list.remove(0);
    assertTrue("list should be empty.", list.isEmpty());
  }

  /**
   * Test method for {@link LinkedList#iterator()}.
   */
  @Test
  public void testIterator() {
    assertNotNull("iterator should not return null.", list.iterator());
  }

  /**
   * Test method for {@link LinkedList#listIterator()}.
   */
  @Test
  public void testListIterator() {
    ListIterator<Integer> it = list.listIterator();
    assertNotNull("iterator should not return null.", it);
    assertEquals("iterator should return 0 as next index.", 0, it.nextIndex());
    list.add(0);
    list.add(1);
    it = list.listIterator();
    assertNotNull("iterator should not return null.", it);
    assertEquals("iterator should return 0 as next index.", 0, it.nextIndex());
  }

  /**
   * Test method for {@link LinkedList#listIterator(int)}.
   */
  @Test
  public void testListIteratorInt() {
    ListIterator<Integer> it = list.listIterator(0);
    assertNotNull("iterator should not return null.", it);
    assertEquals("iterator should return 0 as next index.", 0, it.nextIndex());
    list.add(0);
    list.add(1);
    it = list.listIterator(1);
    assertNotNull("iterator should not return null.", it);
    assertEquals("iterator should return 1 as next index.", 1, it.nextIndex());
    it = list.listIterator(2);
    assertNotNull("iterator should not return null.", it);
    assertEquals("iterator should return 2 as next index.", 2, it.nextIndex());
    try {
      it = list.listIterator(3);
      fail("listIterator should have thrown IndexOutOfBoundsException.");
    } catch (IndexOutOfBoundsException e) {
      // test passed
    }
  }

  /**
   * Test method for {@link LinkedList#toArray()}.
   */
  @Test
  public void testToArray() {
    Object[] array = list.toArray();
    assertNotNull("array should not be null.", array);
    assertEquals("size of array should be 0.", 0, array.length);

    list.add(0);
    list.add(1);
    array = list.toArray();
    assertNotNull("array should not be null.", array);
    assertEquals("size of array should be 2.", 2, array.length);
    assertEquals("position 0 of array should be element 0", 0, array[0]);
    assertEquals("position 1 of array should be element 1", 1, array[1]);
  }

  /**
   * Test method for {@link LinkedList#toArray(T[])}.
   */
  @Test
  public void testToArrayTArray() {
    Integer[] array = list.toArray(new Integer[0]);
    assertNotNull("array should not be null.", array);
    assertEquals("size of array should be 0.", 0, array.length);

    list.add(0);
    list.add(1);
    array = list.toArray(new Integer[0]);
    assertNotNull("array should not be null.", array);
    assertEquals("size of array should be 2.", 2, array.length);
    assertEquals("position 0 of array should be element 0", 0, (int) array[0]);
    assertEquals("position 1 of array should be element 1", 1, (int) array[1]);
  }

  /**
   * Test method for {@link LinkedList#toString()}.
   */
  @Test
  public void testToString() {
    assertEquals("toString should return [].", "[]", list.toString());
    list.add(0);
    list.add(1);
    assertEquals("toString should return [0, 1].", "[0, 1]", list.toString());
  }

  /**
   * Test method for {@link LinkedList#equals(java.lang.Object)}.
   */
  @Test
  public void testEqualsObject() {
    assertTrue("list should be equal to itself.", list.equals(list));
    assertFalse("list should not be equal to null.", list.equals(null));
    assertFalse("list should not be equal to non-list",
        list.equals(new Object()));
    List<Integer> other = new LinkedList<>();
    assertTrue("lists should be equal.", list.equals(other));
    list.add(0);
    other.add(0);
    assertTrue("lists should be equal.", list.equals(other));
    list.add(1);
    assertFalse("lists should not be equal.", list.equals(other));
    list.add(2);
    other.add(2);
    other.add(1);
    assertFalse("lists should not be equal.", list.equals(other));
  }

  /**
   * Test method for {@link java.lang.Object#hashCode()}.
   */
  @Test
  public void testHashCode() {
    assertEquals("hashCode should be 1.", 1, list.hashCode());
    list.addAll(SAMPLE_DATA);
    int sampleDataHashCode = expectedSampleDataHashCode();
    assertEquals("hashCode should be hashCode of sample data.",
        sampleDataHashCode, list.hashCode());
  }

  // --------------------------------------------------
  // ITERATOR TESTS
  // --------------------------------------------------

  /**
   * Test method for {@link LinkedList$LinkedListIterator#LinkedListIterator()}.
   */
  @Test
  public void testLinkedListIteratorLinkedListIterator() {
    ListIterator<Integer> it = list.listIterator();
    assertNotNull("iterator should not return null.", it);
    assertEquals("iterator should return 0 as next index.", 0, it.nextIndex());
  }

  /**
   * Test method for
   * {@link LinkedList$LinkedListIterator#LinkedListIterator(int)}.
   */
  @Test
  public void testLinkedListIteratorLinkedListIteratorInt() {
    list.add(0);
    ListIterator<Integer> it = list.listIterator(1);
    assertNotNull("iterator should not return null.", it);
    assertEquals("iterator should return 1 as next index.", 1, it.nextIndex());
  }

  /**
   * Test method for {@link LinkedList$LinkedListIterator#nextIndex()}.
   */
  @Test
  public void testLinkedListIteratorNextIndex() {
    ListIterator<Integer> it = list.listIterator();
    assertEquals("nextIndex should return 0.", 0, it.nextIndex());
    list.add(0);
    it = list.listIterator();
    assertEquals("nextIndex should return 0.", 0, it.nextIndex());
    it.next();
    assertEquals("nextIndex should return 1.", 1, it.nextIndex());
  }

  /**
   * Test method for {@link LinkedList$LinkedListIterator#hasNext()}.
   */
  @Test
  public void testLinkedListIteratorHasNext() {
    ListIterator<Integer> it = list.listIterator();
    assertFalse("hasNext should return false.", it.hasNext());
    list.add(0);
    it = list.listIterator();
    assertTrue("hasNext should return true.", it.hasNext());
    it.next();
    assertFalse("hasNext should return false.", it.hasNext());
  }

  /**
   * Test method for {@link LinkedList$LinkedListIterator#next()}.
   */
  @Test
  public void testLinkedListIteratorNext() {
    ListIterator<Integer> it = list.listIterator();
    try {
      it.next();
      fail("next should have thrown NoSuchElementException.");
    } catch (NoSuchElementException e) {
      // test passed
    }
    list.add(0);
    list.add(1);
    it = list.listIterator();
    assertEquals("next should return element 0.", 0, (int) it.next());
    assertEquals("next should return element 1.", 1, (int) it.next());
    try {
      it.next();
      fail("next should have thrown NoSuchElementException.");
    } catch (NoSuchElementException e) {
      // test passed
    }
    it.previous();
    assertEquals("next should return element 1.", 1, (int) it.next());
    try {
      it.next();
      fail("next should have thrown NoSuchElementException.");
    } catch (NoSuchElementException e) {
      // test passed
    }
  }

  /**
   * Test method for {@link LinkedList$LinkedListIterator#previousIndex()}.
   */
  @Test
  public void testLinkedListIteratorPreviousIndex() {
    ListIterator<Integer> it = list.listIterator();
    assertEquals("previousIndex should return -1.", -1, it.previousIndex());
    list.add(0);
    it = list.listIterator();
    assertEquals("previousIndex should return -1.", -1, it.previousIndex());
    it.next();
    assertEquals("previousIndex should return 0.", 0, it.previousIndex());
  }

  /**
   * Test method for {@link LinkedList$LinkedListIterator#hasPrevious()}.
   */
  @Test
  public void testLinkedListIteratorHasPrevious() {
    ListIterator<Integer> it = list.listIterator();
    assertFalse("hasPrevious should return false.", it.hasPrevious());
    list.add(0);
    it = list.listIterator();
    assertFalse("hasPrevious should return false.", it.hasPrevious());
    it.next();
    assertTrue("hasPrevious should return true.", it.hasPrevious());
  }

  /**
   * Test method for {@link LinkedList$LinkedListIterator#previous()}.
   */
  @Test
  public void testLinkedListIteratorPrevious() {
    ListIterator<Integer> it = list.listIterator();
    try {
      it.previous();
      fail("previous should have thrown NoSuchElementException.");
    } catch (NoSuchElementException e) {
      // test passed
    }
    list.add(0);
    list.add(1);
    it = list.listIterator();
    try {
      it.previous();
      fail("previous should have thrown NoSuchElementException.");
    } catch (NoSuchElementException e) {
      // test passed
    }
    it.next();
    assertEquals("previous should return element 0.", 0, (int) it.previous());
    try {
      it.previous();
      fail("previous should have thrown NoSuchElementException.");
    } catch (NoSuchElementException e) {
      // test passed
    }
    it.next();
    assertEquals("previous should return element 0.", 0, (int) it.previous());
    it.next();
    it.next();
    assertEquals("previous should return element 1.", 1, (int) it.previous());
  }

  /**
   * Test method for {@link LinkedList$LinkedListIterator#add(java.lang.Object)}
   * .
   */
  @Test
  public void testLinkedListIteratorAdd() {
    ListIterator<Integer> it = list.listIterator();
    it.add(0);
    assertEquals("size should be 1.", 1, list.size());
    assertEquals("nextIndex should return 1.", 1, it.nextIndex());
    assertFalse("hasNext should return false.", it.hasNext());
    it.add(1);
    assertEquals("size should be 2.", 2, list.size());
    assertEquals("nextIndex should return 2.", 2, it.nextIndex());
    assertFalse("hasNext should return false.", it.hasNext());
    assertEquals("previous element should be element 1.", 1,
        (int) it.previous());
    assertEquals("previous element should be element 0.", 0,
        (int) it.previous());
    it.next();
    it.add(2);
    assertEquals("size should be 3.", 3, list.size());
    assertEquals("nextIndex should return 2.", 2, it.nextIndex());
    assertTrue("hasNext should return true.", it.hasNext());
    assertEquals("element after added element should be element 1.", 1,
        (int) it.next());
    it.previous();
    it.previous();
    assertEquals("element before added element should be element 0.", 0,
        (int) it.previous());
  }

  /**
   * Test method for {@link LinkedList$LinkedListIterator#set(java.lang.Object)}
   * .
   */
  @Test
  public void testLinkedListIteratorSet() {
    ListIterator<Integer> it = list.listIterator();
    try {
      it.set(0);
      fail("set should have thrown IllegalStateException.");
    } catch (IllegalStateException e) {
      // test passed
    }
    list.add(0);
    it = list.listIterator();
    try {
      it.set(1);
      fail("set should have thrown IllegalStateException.");
    } catch (IllegalStateException e) {
      // test passed
    }
    it.next();
    it.set(1);
    assertEquals("size should be 1.", 1, list.size());
    assertEquals("nextIndex should return 1.", 1, it.nextIndex());
    assertEquals("element should have been set to 1.", 1, (int) it.previous());
    it.set(2);
    assertEquals("size should be 1.", 1, list.size());
    assertEquals("nextIndex should return 1.", 0, it.nextIndex());
    assertEquals("element should have been set to 2.", 2, (int) it.next());
    try {
      it.add(0);
      it.set(1);
      fail("set should have thrown IllegalStateException.");
    } catch (IllegalStateException e) {
      // test passed
    }
  }

  /**
   * Test method for {@link LinkedList$LinkedListIterator#remove()} .
   */
  @Test
  public void testLinkedListIteratorRemove() {
    ListIterator<Integer> it = list.listIterator();
    try {
      it.remove();
      fail("remove should have thrown IllegalStateException.");
    } catch (IllegalStateException e) {
      // test passed
    }
    list.add(0);
    it = list.listIterator();
    try {
      it.remove();
      fail("remove should have thrown IllegalStateException.");
    } catch (IllegalStateException e) {
      // test passed
    }
    it.next();
    it.remove();
    assertEquals("size should be 0.", 0, list.size());
    assertEquals("nextIndex should return 0.", 0, it.nextIndex());
    list.add(0);
    it = list.listIterator();
    it.next();
    it.previous();
    it.remove();
    assertEquals("size should be 0.", 0, list.size());
    assertEquals("nextIndex should return 0.", 0, it.nextIndex());
    list.add(0);
    list.add(1);
    list.add(2);
    it = list.listIterator();
    it.next();
    it.next();
    it.remove();
    assertEquals("size should be 2.", 2, list.size());
    assertEquals("nextIndex should return 1.", 1, it.nextIndex());
    assertEquals("element after removed element should be element 2.", 2,
        (int) it.next());
    it.previous();
    assertEquals("element before removed element should be element 0.", 0,
        (int) it.previous());
  }
}
